package computercraftsc.client.gui.container;

import computercraftsc.client.gui.container.slot.ProgrammingSlot;
import computercraftsc.client.gui.container.slot.TurtleSlot;
import computercraftsc.client.gui.inventories.InventoryManager;
import computercraftsc.client.gui.inventories.config.CodeItemEntry;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import computercraftsc.client.gui.GuiConstants;
import computercraftsc.client.gui.container.TurtleContainerSC.GUIPopupHandler.PopupType;
import computercraftsc.client.gui.container.slot.ButtonSlot;
import computercraftsc.client.gui.container.slot.CodingSlot;
import computercraftsc.client.gui.container.slotmanager.SlotManager;
import computercraftsc.shared.RegistrySC;
import computercraftsc.shared.items.ItemBlockName;
import computercraftsc.shared.items.ItemComment;
import computercraftsc.shared.items.ItemItemName;
import computercraftsc.shared.items.ItemNumber;
import computercraftsc.shared.items.ItemProgrammingIcon;
import computercraftsc.shared.items.ItemString;
import computercraftsc.shared.items.ItemVariable;
import computercraftsc.shared.turtle.block.TileTurtleSC;
import net.minecraft.client.Minecraft;
import net.minecraft.entity.player.PlayerEntity;
import net.minecraft.entity.player.PlayerInventory;
import net.minecraft.inventory.container.ClickType;
import net.minecraft.inventory.container.Container;
import net.minecraft.inventory.container.Slot;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.network.PacketBuffer;
import net.minecraft.tileentity.TileEntity;

public class TurtleContainerSC extends Container {

	private static final Logger LOGGER = LogManager.getLogger();

	private final TileTurtleSC tileTurtleSC;
	private final InventoryManager inventoryManager;
	private final SlotManager slotManager;
	private GUIPopupHandler popupHandler = null;

	/**
	 * This Class holds most of the Slot Logic for the turtle.
	 * Big parts of the logic are delegated to specialized classes
	 *
	 * @param playerInv    Inventory of the player interacting with the turtle
	 * @param tileTurtleSC The TileEntity of the turtle.
	 */
	public TurtleContainerSC(int windowId, PlayerInventory playerInv, TileTurtleSC tileTurtleSC) {
		super(RegistrySC.ModContainers.TURTLE_CONTAINER_SC.get(), windowId);
		this.tileTurtleSC = tileTurtleSC;
		this.inventoryManager = tileTurtleSC.getInventoryManager();
		this.inventoryManager.setInventoryPlayer(playerInv); // We need the Inventory Player to display the Players Inventory in the Turtle.
		this.slotManager = new SlotManager(this, this.inventoryManager, this.tileTurtleSC); // The SlotManager is responsible for placing the slots on the screen
	}

	public static TurtleContainerSC createContainerClientSide(
			int windowID, PlayerInventory playerInv, PacketBuffer extraData) {
		@SuppressWarnings("resource")
		TileEntity tileEntity = Minecraft.getInstance().world.getTileEntity(extraData.readBlockPos());
		if(!(tileEntity instanceof TileTurtleSC)) {
			LOGGER.error("Attempting to create client side " + TurtleContainerSC.class.getSimpleName()
					+ " for a position without a " + TileTurtleSC.class.getSimpleName());
			return null;
		}
		TileTurtleSC tileTurtleSC = (TileTurtleSC) tileEntity;

		// Set config mode.
		tileTurtleSC.getInventoryManager().setConfigMode(extraData.readBoolean());

		// Set inventory lock states.
		tileTurtleSC.setInventoriesLocked(extraData.readBoolean());
		tileTurtleSC.setTurtleInventoryLocked(extraData.readBoolean());

		// Set code item entries.
		int numCodeItemEntries = extraData.readInt();
		List<CodeItemEntry> codeItemEntries = new ArrayList<>(numCodeItemEntries);
		for(int i = 0; i < numCodeItemEntries; i++) {
			codeItemEntries.add(CodeItemEntry.readFromNBT(extraData.readCompoundTag()));
		}
		Collections.sort(codeItemEntries);
		tileTurtleSC.getInventoryManager().setCodeItemEntries(codeItemEntries);

		// Set infinite code items setting.
		tileTurtleSC.getInventoryManager().getProgrammingInventory().setHasInfiniteCodeItems(extraData.readBoolean());

		// Set locked code inventory slots.
		int numLockedCodingSlotEntries = extraData.readInt();
		Set<Integer> lockedCodingSlotInvIds = tileTurtleSC
				.getInventoryManager().getCodingInventory().getLockedInventorySlotIds();
		lockedCodingSlotInvIds.clear();
		for(int i = 0; i < numLockedCodingSlotEntries; i++) {
			lockedCodingSlotInvIds.add(extraData.readInt());
		}

		// Return the new container.
		return new TurtleContainerSC(windowID, playerInv, tileTurtleSC);
	}

	public SlotManager getSlotManager() {
		return this.slotManager;
	}

	public TileTurtleSC getTileTurtleSC() {
		return this.tileTurtleSC;
	}

	public void setPopupHandler(GUIPopupHandler popupHandler) {
		this.popupHandler = popupHandler;
	}

	@Override
	public void onContainerClosed(PlayerEntity player) {

		// Clear the item held by the player, dropping it if it is not a code item.
		if (!player.inventory.getItemStack().isEmpty()) {
			if (!(player.inventory.getItemStack().getItem() instanceof ItemProgrammingIcon)) {
				player.dropItem(player.inventory.getItemStack(), true);
			}
			player.inventory.setItemStack(ItemStack.EMPTY);
		}

		// Notify the tile entity about the closing.
		this.tileTurtleSC.closeInventory(player);
	}

	@Override
	public boolean canInteractWith(PlayerEntity playerIn) {
		return this.tileTurtleSC.isUsableByPlayer(playerIn);
	}

	/**
	 * Handles clicks in the GUI. This implementation varies from vanilla behavior in the following points:
	 * <ul>
	 * <li>Forwards {@link ButtonSlot} clicks to their {@link ButtonSlot#onSlotClicked()} handler.</li>
	 * <li>Opens a GUI popup when a configurable {@link ItemProgrammingIcon} is placed in a {@link CodingSlot}.</li>
	 * <li>Limits item stack modifications in {@link ProgrammingSlot}s to changing their stack amounts (>= 0).</li>
	 * <li>Moves {@link ItemProgrammingIcon}s placed in any {@link ProgrammingSlot} to the {@link ProgrammingSlot}
	 * in which they belong.</li>
	 * </ul>
	 * @param slotId - Slot index.
	 * @param clickedButton - Clicked button (see https://wiki.vg/Protocol#Click_Window).
	 * @param clickType - Click mode (see https://wiki.vg/Protocol#Click_Window). Click types match modes as follows:
	 * {0: PICKUP, 1: QUICK_MOVE, 2: SWAP, 3: CLONE, 4: THROW, 5: QUICK_CRAFT, 6: PICKUP_ALL}.
	 * @param playerIn - The clicking player.
	 * @return The clicked {@link ItemStack} that should end up on the mouse (this is an educated guess, often unused).
	 */
	@Override
	public ItemStack slotClick(int slotId, int clickedButton, ClickType clickType, PlayerEntity player) {
		PlayerInventory pinv = player.inventory;

		// Move items to the programming inventory when they are dropped outside of the GUI.
		if (slotId == -999 && (clickType == ClickType.PICKUP || clickType == ClickType.QUICK_MOVE)
				&& (clickedButton == 0 || clickedButton == 1)) {
			ItemStack heldItemStack = pinv.getItemStack();
			if (heldItemStack.getItem() instanceof ItemProgrammingIcon) {
				if (clickedButton == 0) {

					// Left click: Discard all held items.
					pinv.setItemStack(ItemStack.EMPTY);
					this.inventoryManager.getProgrammingInventory().returnItemStack(heldItemStack);
				} else {

					// Right click: Discard one held item.
					Item heldItem = heldItemStack.getItem();
					pinv.getItemStack().shrink(1);
					if (pinv.getItemStack().isEmpty()) {
						pinv.setItemStack(ItemStack.EMPTY);
					}
					this.inventoryManager.getProgrammingInventory()
							.returnItemStack(new ItemStack(heldItem, 1));
				}
				return ItemStack.EMPTY;
			}
		}

		// Pass remaining clicks outside of the GUI (slotId -999), drag start/stops (slotId -999)
		// and on the GUI's edge (slotId -1) to the super class.
		if (slotId < 0) {
			return super.slotClick(slotId, clickedButton, clickType, player);
		}

		// Ignore invalid slot ids (Clients can send any integer).
		if(slotId > this.inventorySlots.size()) {
			return ItemStack.EMPTY;
		}

		// Get the clicked slot.
		Slot slot = this.inventorySlots.get(slotId);

		// Pass button slot clicks to their handler.
		if (slot instanceof ButtonSlot) {
			((ButtonSlot) slot).onSlotClicked(clickType, clickedButton, player);
			return ItemStack.EMPTY;
		}

		// Disable inventory slot clicks in non-config mode if all inventories are locked.
		if (this.tileTurtleSC.getInventoriesLocked() && !this.inventoryManager.isConfigMode()) {
			return ItemStack.EMPTY;
		}

		// Disable turtle inventory slot clicks in non-config mode if that inventory is locked.
		if (slot instanceof TurtleSlot
				&& this.tileTurtleSC.getTurtleInventoryLocked() && !this.inventoryManager.isConfigMode()) {
			return ItemStack.EMPTY;
		}

		// Move items to the programming inventory when they are dropped using Q or CTRL+Q.
		if (clickType == ClickType.THROW) {
			if (!slot.getStack().isEmpty() && slot.getStack().getItem() instanceof ItemProgrammingIcon) {

				// Disallow programming inventory items to be dropped.
				if (slot instanceof ProgrammingSlot) {
					return ItemStack.EMPTY;
				}

				// Disallow locked coding inventory slot items to be dropped.
				if(slot instanceof CodingSlot && this.inventoryManager
						.getCodingInventory().getLockedInventorySlotIds().contains(slot.getSlotIndex())) {
					return ItemStack.EMPTY;
				}

				ItemStack slotItemStack = slot.getStack();
				if (clickedButton == 0) {

					// Q pressed: Discard one item.
					Item slotItemStackItem = slotItemStack.getItem();
					slotItemStack.shrink(1);
					if(slotItemStack.isEmpty()) {
						slot.putStack(ItemStack.EMPTY);
					}
					this.inventoryManager.getProgrammingInventory()
							.returnItemStack(new ItemStack(slotItemStackItem, 1));
				} else if (clickedButton == 1) {

					// CTRL+Q pressed: Discard all items.
					slot.putStack(ItemStack.EMPTY);
					this.inventoryManager.getProgrammingInventory().returnItemStack(slotItemStack);
				}
				return ItemStack.EMPTY;
			}
		}

		// Handle coding inventory clicks.
		if (slot instanceof CodingSlot) {

			// Disallow non-empty locked coding inventory slots to be modified in non-config mode.
			boolean isConfigMode = this.inventoryManager.isConfigMode();
			if(!isConfigMode && !slot.getStack().isEmpty() && this.inventoryManager
					.getCodingInventory().getLockedInventorySlotIds().contains(slot.getSlotIndex())) {
				return ItemStack.EMPTY;
			}

			switch (clickType) {

				// Left/right click coding inventory.
				case PICKUP_ALL: {

					// This is a double click, so perform one recursive normal click and handle the other one below.
					this.slotClick(slotId, clickedButton, ClickType.PICKUP, player);
					// Intended fallthrough.
				}
				case PICKUP: {
					switch (clickedButton) {

						// Left click coding inventory: Place/merge whole stack. Swap if possible. Do nothing otherwise.
						case 0: {

							// Handle pick up full stack with empty hand.
							if (pinv.getItemStack().isEmpty()) {
								if (!slot.getStack().isEmpty()) {
									pinv.setItemStack(slot.getStack());
									slot.putStack(ItemStack.EMPTY);
								}
								return ItemStack.EMPTY;
							}

							// Handle place full stack in empty slot.
							if (slot.getStack().isEmpty()) {
								slot.putStack(pinv.getItemStack());
								pinv.setItemStack(ItemStack.EMPTY);

								// Open input popup when a customizable item will be placed in the coding inventory.
								this.openInputPopup(slot);

								return ItemStack.EMPTY;
							}

							// Handle swap/merge stacks with item in slot and in hand.
							if (pinv.getItemStack().getItem().equals(slot.getStack().getItem())) {

								// Merge same items if possible (placing as many as possible).
								int maxExtraItems = slot.getStack().getMaxStackSize() - slot.getStack().getCount();
								int placeAmount = (pinv.getItemStack().getCount() > maxExtraItems
										? maxExtraItems : pinv.getItemStack().getCount());
								if (placeAmount > 0) {
									pinv.getItemStack().shrink(placeAmount);
									if (pinv.getItemStack().isEmpty()) {
										pinv.setItemStack(ItemStack.EMPTY);
									}
									slot.getStack().grow(placeAmount);
									slot.onSlotChanged();
								}
							} else {

								// Swap stacks if possible.
								ItemStack heldItemStack = pinv.getItemStack();
								if (heldItemStack.getCount() <= heldItemStack.getMaxStackSize()) {
									pinv.setItemStack(slot.getStack());
									slot.putStack(heldItemStack);

									// Open input popup when a customizable item will be placed in the coding inventory.
									this.openInputPopup(slot);
								}
							}
							return ItemStack.EMPTY;
						}

						// Right click coding inventory: Place 1 item if possible.
						// Right click coding inventory in config mode: Toggle coding inventory slot lock state.
						case 1: {

							// Handle right click with empty hand in config mode.
							if(isConfigMode && pinv.getItemStack().isEmpty()) {

								// Toggle coding inventory slot lock state.
								Set<Integer> lockedInventorySlotIds = this.inventoryManager
										.getCodingInventory().getLockedInventorySlotIds();
								if(!lockedInventorySlotIds.remove(slot.getSlotIndex())) {
									lockedInventorySlotIds.add(slot.getSlotIndex());
								}
								return ItemStack.EMPTY;
							}

							if (!pinv.getItemStack().isEmpty()) {

								// Disallow placement on a stack of another type or with no space left.
								if (!slot.getStack().isEmpty()
										&& (!slot.getStack().getItem().equals(pinv.getItemStack().getItem())
										|| slot.getStack().getCount() >= slot.getStack().getMaxStackSize())) {
									return ItemStack.EMPTY;
								}

								// Place one item.
								if (slot.getStack().isEmpty()) {
									slot.putStack(pinv.getItemStack().split(1));
								} else {
									pinv.getItemStack().shrink(1);
									slot.getStack().grow(1);
									slot.onSlotChanged();
								}
								if (pinv.getItemStack().isEmpty()) {
									pinv.setItemStack(ItemStack.EMPTY);
								}

								// Open input popup when a customizable item will be placed in the coding inventory.
								this.openInputPopup(slot);

								return ItemStack.EMPTY;
							}
							return ItemStack.EMPTY;
						}

						default: {
							throw new Error();
						}
					}
				}

				// Shift + click coding inventory: Move item to programming inventory.
				// Shift + right click with empty hand in config mode: Toggle multi coding inventory slot lock state.
				case QUICK_MOVE: {

					// Handle shift + right click with empty hand in config mode.
					if(clickedButton == 1 && isConfigMode && pinv.getItemStack().isEmpty()) {

						// Get lock/unlock operation.
						Set<Integer> lockedInventorySlotIds = this.inventoryManager
								.getCodingInventory().getLockedInventorySlotIds();
						boolean lock = !lockedInventorySlotIds.contains(slot.getSlotIndex());

						// Toggle all slot lock states until the previous slot with a different lock state.
						int i = slot.getSlotIndex();
						do {
							if((lock && !lockedInventorySlotIds.add(i))
									|| (!lock && !lockedInventorySlotIds.remove(i))) {
								break;
							}
						} while(--i >= 0);
						return ItemStack.EMPTY;
					}

					ItemStack itemStack = slot.getStack();
					if (!slot.getStack().isEmpty()) {
						slot.putStack(ItemStack.EMPTY);
						this.inventoryManager.getProgrammingInventory().returnItemStack(itemStack);
					}
					return ItemStack.EMPTY;
				}

				// Drag coding inventory (either mouse button): Place 1 item if the dragged slot allows it.
				case QUICK_CRAFT: {

					// Return if no stack is being held.
					ItemStack heldItemStack = pinv.getItemStack();
					if (heldItemStack.isEmpty()) {
						return ItemStack.EMPTY;
					}

					// Handle place non-empty stack in empty slot (placing one item).
					if (slot.getStack().isEmpty()) {
						slot.putStack(heldItemStack.split(1));
						if (heldItemStack.isEmpty()) {
							pinv.setItemStack(ItemStack.EMPTY);
						}

						// Open input popup when a customizable item will be placed in the coding inventory.
						this.openInputPopup(slot);

						return ItemStack.EMPTY;
					}

					// Handle merge stacks with item in slot and in hand.
					if (heldItemStack.getItem().equals(slot.getStack().getItem())) {

						// Merge same items if possible (placing one item).
						int maxExtraItems = slot.getStack().getMaxStackSize() - slot.getStack().getCount();
						if (maxExtraItems > 0) {
							heldItemStack.shrink(1);
							if (heldItemStack.isEmpty()) {
								pinv.setItemStack(ItemStack.EMPTY);
							}
							slot.getStack().grow(1);
							slot.onSlotChanged();
						}
					}

					// Do nothing when dragging over a slot in which the held item cannot be placed.
					return ItemStack.EMPTY;
				}

				// Disallow number keys (mode 2), middle-mouse creative clone (mode 3), Q item dropping (mode 4),
				// item dragging (mode 5) and double click collect-all (mode 6) in the coding inventory.
				default: {
					return ItemStack.EMPTY;
				}
			}
		}

		// Handle programming inventory clicks.
		if (slot instanceof ProgrammingSlot) {
			switch (clickType) {

				// Left/right click programming inventory.
				case PICKUP_ALL: {

					// This is a double click, so perform one recursive normal click and handle the other one below.
					this.slotClick(slotId, clickedButton, ClickType.PICKUP, player);
					// Intended fallthrough.
				}
				case PICKUP: {
					switch (clickedButton) {

						// Left click programming inventory:
						// Take +1 item if possible, dumping held item if it's different or if the slot is empty.
						case 0: {

							// Dump held item if no items are available in the clicked slot.
							ItemStack heldItemStack = pinv.getItemStack();
							if (slot.getStack().isEmpty()) {
								if (!heldItemStack.isEmpty()) {
									pinv.setItemStack(ItemStack.EMPTY);
									this.inventoryManager.getProgrammingInventory().returnItemStack(heldItemStack);
								}
								return ItemStack.EMPTY;
							}

							// Take +1 item if possible, dumping the held item if it's different.
							if (heldItemStack.isEmpty() || !heldItemStack.getItem().equals(slot.getStack().getItem())) {
								pinv.setItemStack(new ItemStack(slot.getStack().getItem(), 1));
								pinv.setItemStack(slot.decrStackSize(1));
								if (!heldItemStack.isEmpty()) {
									this.inventoryManager.getProgrammingInventory().returnItemStack(heldItemStack);
								}
							} else {
								if (heldItemStack.getCount() < heldItemStack.getMaxStackSize()) {
									heldItemStack.grow(1);
									slot.getStack().shrink(1);
									slot.onSlotChanged();
								}
							}
							return ItemStack.EMPTY;
						}

						// Right click programming inventory: Dump -1 item if possible.
						case 1: {
							ItemStack heldItemStack = pinv.getItemStack();
							if (!heldItemStack.isEmpty()) {
								Item heldItem = heldItemStack.getItem();
								heldItemStack.shrink(1);
								if (heldItemStack.isEmpty()) {
									pinv.setItemStack(ItemStack.EMPTY);
								}
								this.inventoryManager.getProgrammingInventory()
										.returnItemStack(new ItemStack(heldItem, 1));
							}
							return ItemStack.EMPTY;
						}

						default: {
							throw new Error();
						}
					}
				}

				// Shift + click programming inventory: Move item to coding inventory, attempting a smart insert.
				case QUICK_MOVE: {

					// Take the item stack from the programming slot.
					ItemStack itemStack = slot.decrStackSize(1);

					// Insert the item in a smart position in the coding inventory.
					int placedInventoryIndex = this.inventoryManager
							.getCodingInventory().appendStackWithSmartIndentation(itemStack);
					if (placedInventoryIndex == -1) {

						// Restore the programming inventory stack.
						if (slot.getStack().isEmpty()) {
							slot.putStack(itemStack);
						} else {
							slot.getStack().grow(itemStack.getCount());
							slot.onSlotChanged();
						}
					} else {

						// Scroll to the inventory slot in which the item was placed if it is outside of the view.
						if(player.world.isRemote) {
							this.slotManager.getCodingInventorySlotManager().scrollSlotToView(placedInventoryIndex);
						}

						// Open input popup when a customizable item will be placed in the coding inventory.
						Slot placedSlot = this.slotManager
								.getCodingInventorySlotManager().getSlot(placedInventoryIndex);
						if (placedSlot instanceof CodingSlot) {
							this.openInputPopup(placedSlot);
						}
					}
					return ItemStack.EMPTY;
				}

				// Disallow number keys (mode 2), middle-mouse creative clone (mode 3), Q item dropping (mode 4),
				// item dragging (mode 5) and double click collect-all (mode 6) in the coding inventory.
				default: {
					return ItemStack.EMPTY;
				}
			}
		}

		// Disable shift-clicking, number key presses, and double clicking in all remaining inventories.
		if (clickType == ClickType.QUICK_MOVE || clickType == ClickType.SWAP || clickType == ClickType.PICKUP_ALL) {
			return ItemStack.EMPTY;
		}

		// Handle vanilla slot click behavior.
		return super.slotClick(slotId, clickedButton, clickType, player);
	}

	/**
	 * Opens an input popup when a customizable item is placed in an inventory slot.
	 * Does nothing if the turtle is in config mode, or when no popup handler is set.
	 * @param placedSlot - The {@link Slot} in which the possibly customizable item was placed.
	 */
	public void openInputPopup(Slot placedSlot) {
		ItemStack placedItemStack = placedSlot.getStack();
		popupSelect:
		if(!this.inventoryManager.isConfigMode()
				&& placedItemStack.getItem() instanceof ItemProgrammingIcon && this.popupHandler != null) {
			ItemProgrammingIcon item = (ItemProgrammingIcon) placedItemStack.getItem();
			PopupType type;
			if(item instanceof ItemNumber) {
				type = PopupType.NUMBER;
			} else if(item instanceof ItemVariable) {
				type = PopupType.VARIABLE;
			} else if(item instanceof ItemString || item instanceof ItemComment) {
				type = PopupType.STRING;
			} else if(item instanceof ItemBlockName) {
				type = PopupType.BLOCK_NAME;
			} else if(item instanceof ItemItemName) {
				type = PopupType.ITEM_NAME;
			} else {
				break popupSelect;
			}
			this.popupHandler.showPopup(type, placedSlot.xPos,
					placedSlot.yPos + GuiConstants.SLOT_SIZE, placedSlot.slotNumber);
		}
	}

	@Override
	public Slot addSlot(Slot slot) {
		return super.addSlot(slot);
	}

	public interface GUIPopupHandler {

		/**
		 * Opens a popup of the given {@link PopupType}, and puts the value inserted by the user into the item stack at
		 * the given slot index.
		 * @param type - The popup type.
		 * @param popupX - The x coordinate of the left side of the popup.
		 * @param popupY - The y coordinate of the top of the popup.
		 * @param slot - The slot index in the container.
		 */
		void showPopup(PopupType type, int popupX, int popupY, int slot);

		enum PopupType {
			VARIABLE,
			NUMBER,
			STRING,
			BLOCK_NAME,
			ITEM_NAME;
		}
	}
}
