package computercraftsc.client.gui.container;

import computercraftsc.SCGlobal;
import computercraftsc.client.gui.GuiConstants;
import computercraftsc.client.gui.KeyCodes;
import computercraftsc.client.gui.TurtleInventoryScreen;
import computercraftsc.client.gui.container.TurtleContainerSC.GUIPopupHandler;
import computercraftsc.client.gui.container.slot.ButtonSlot;
import computercraftsc.client.gui.container.slot.CodingSlot;
import computercraftsc.client.gui.container.slot.ProgrammingSlot;
import computercraftsc.client.gui.widgets.IBlockSelectionListener;
import computercraftsc.client.gui.widgets.IItemSelectionListener;
import computercraftsc.client.gui.widgets.WidgetBlockSelectionPopup;
import computercraftsc.client.gui.widgets.WidgetContainer;
import computercraftsc.client.gui.widgets.WidgetItemSelectionPopup;
import computercraftsc.client.gui.widgets.WidgetManager;
import computercraftsc.client.gui.widgets.WidgetPopup;
import computercraftsc.client.gui.widgets.WidgetTextEntryPopup;
import computercraftsc.client.gui.widgets.WidgetTextEntryPopup.TextType;
import computercraftsc.shared.items.ItemBlockName;
import computercraftsc.shared.items.ItemComment;
import computercraftsc.shared.items.ItemGuiButton;
import computercraftsc.shared.items.ItemItemName;
import computercraftsc.shared.items.ItemNumber;
import computercraftsc.shared.items.ItemProgrammingIcon;
import computercraftsc.shared.items.ItemString;
import computercraftsc.shared.items.ItemVariable;
import computercraftsc.shared.network.NetworkManagerSC;
import computercraftsc.shared.network.packet.SetCodingSlotStackPacket;
import computercraftsc.shared.turtle.block.TileTurtleSC;
import computercraftsc.shared.turtle.core.code.compiler.CompileResult;
import computercraftsc.shared.turtle.core.code.compiler.exception.CompileException;
import net.minecraft.block.Block;
import net.minecraft.client.gui.AbstractGui;
import net.minecraft.client.gui.FontRenderer;
import net.minecraft.client.gui.screen.inventory.ContainerScreen;
import net.minecraft.client.renderer.IRenderTypeBuffer;
import net.minecraft.client.renderer.RenderHelper;
import net.minecraft.client.renderer.Tessellator;
import net.minecraft.client.renderer.texture.NativeImage;
import net.minecraft.client.renderer.texture.SimpleTexture.TextureData;
import net.minecraft.entity.player.PlayerInventory;
import net.minecraft.inventory.container.ClickType;
import net.minecraft.inventory.container.Container;
import net.minecraft.inventory.container.Slot;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.util.ResourceLocation;
import net.minecraft.util.math.vector.Matrix4f;
import net.minecraft.util.text.IFormattableTextComponent;
import net.minecraft.util.text.ITextComponent;
import net.minecraft.util.text.ITextProperties;
import net.minecraft.util.text.LanguageMap;
import net.minecraft.util.text.StringTextComponent;
import net.minecraft.util.text.Style;
import net.minecraft.util.text.TextFormatting;
import net.minecraft.util.text.TranslationTextComponent;
import net.minecraftforge.api.distmarker.Dist;
import net.minecraftforge.api.distmarker.OnlyIn;
import net.minecraftforge.fml.client.gui.GuiUtils;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.mojang.blaze3d.matrix.MatrixStack;
import com.mojang.blaze3d.systems.RenderSystem;

import java.awt.Color;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nonnull;

import static computercraftsc.client.gui.GuiConstants.BACKGROUND_IMAGE_HEIGHT;
import static computercraftsc.client.gui.GuiConstants.BACKGROUND_IMAGE_WIDTH;
import static computercraftsc.client.gui.GuiConstants.SLOT_SIZE;
import static computercraftsc.client.gui.GuiConstants.GUI_SCREEN_NAME_Y_OFFSET;
import static computercraftsc.client.gui.GuiConstants.TURTLE_INVENTORY_HEIGHT;
import static computercraftsc.client.gui.GuiConstants.TURTLE_INVENTORY_WIDTH;
import static computercraftsc.client.gui.GuiConstants.TURTLE_INVENTORY_X_OFFSET;
import static computercraftsc.client.gui.GuiConstants.X_OFFSET_LEFT;
import static computercraftsc.client.gui.GuiConstants.Y_OFFSET_TOP;

@OnlyIn(Dist.CLIENT)
public class TurtleGuiContainerSC extends ContainerScreen<TurtleContainerSC>
		implements GUIPopupHandler, GuiScreenHandler {

    private static final Logger LOGGER = LogManager.getLogger();

	private final TileTurtleSC tileTurtleSC;
	private WidgetManager widgetManager;
	private WidgetPopup popup = null;

	public TurtleGuiContainerSC(TurtleContainerSC turtleContainerSC, PlayerInventory playerInv, ITextComponent title) {
		super(turtleContainerSC, playerInv, title);

		this.tileTurtleSC = turtleContainerSC.getTileTurtleSC();
		this.container.getSlotManager().setGuiScreenHandler(this);
		this.container.setPopupHandler(this);
		this.tileTurtleSC.getInventoryManager().getProgrammingInventory().setContainerHandler(this);

		// Set screen dimensions.
		this.xSize = BACKGROUND_IMAGE_WIDTH;
		this.ySize = BACKGROUND_IMAGE_HEIGHT;
	}

	@Override
	public void init() {
		super.init();
		this.widgetManager = new WidgetManager(this.guiLeft, this.guiTop, this.tileTurtleSC.getInventoryManager(),
				this.container.getSlotManager());

		// Enable keyboard repeat events.
		this.minecraft.keyboardListener.enableRepeatEvents(true);
	}

	@Override
	public void onClose() {
		super.onClose();

		// Disable keyboard repeat events.
		this.minecraft.keyboardListener.enableRepeatEvents(false);
	}

	@Override
	public void render(MatrixStack matrixStack, int mouseX, int mouseY, float partialTicks) {
		super.render(matrixStack, mouseX, mouseY, partialTicks);

		// Draw custom item stack overlays.
		for(int i = 0; i < this.container.inventorySlots.size(); i++) {
			Slot slot = this.container.inventorySlots.get(i);
			if(slot.isEnabled()) {
				this.renderCustomItemStackOverlay(slot.getStack(), this.guiLeft + slot.xPos, this.guiTop + slot.yPos);
			}
		}

		// Draw hovered item stack tooltip.
		RenderSystem.color4f(1.0f, 1.0f, 1.0f, 1.0f);
		this.renderHoveredTooltip(matrixStack, mouseX, mouseY);

		// Blur the turtle item inventory if it is locked.
		if (this.tileTurtleSC.getTurtleInventoryLocked() && !this.tileTurtleSC.getInventoryManager().isConfigMode()
				&& this.widgetManager.getScreen() == TurtleInventoryScreen.INVENTORY_SCREEN) {
			RenderSystem.pushMatrix();
			RenderSystem.translatef(this.guiLeft + TURTLE_INVENTORY_X_OFFSET, this.guiTop + Y_OFFSET_TOP, 1000f);

			this.fillGradient(matrixStack, 0, 0, TURTLE_INVENTORY_WIDTH, TURTLE_INVENTORY_HEIGHT,
					new Color(16, 16, 16, 160).getRGB(), new Color(16, 16, 16, 160).getRGB());

			RenderSystem.popMatrix();
		}

		// Blur locked coding inventory slots.
		Set<Integer> lockedCodingInvSlotIds = this.tileTurtleSC
				.getInventoryManager().getCodingInventory().getLockedInventorySlotIds();
		if(!lockedCodingInvSlotIds.isEmpty()) {
			RenderSystem.pushMatrix();
			RenderSystem.translatef(this.guiLeft, this.guiTop, 350f); // Z of 300 to 400 works with items and popups.

			for(Slot slot : this.container.inventorySlots) {
				if(slot instanceof CodingSlot && lockedCodingInvSlotIds.contains(slot.getSlotIndex())
						&& slot.xPos != GuiConstants.INVISIBLE_LOCATION
						&& slot.yPos != GuiConstants.INVISIBLE_LOCATION) {
					int color = new Color(50, 50, 50, 160).getRGB();
					this.fillGradient(matrixStack, slot.xPos, slot.yPos, slot.xPos + 16, slot.yPos + 16, color, color);
				}
			}

			RenderSystem.popMatrix();
		}

		// Draw the popup if a popup is active.
		if(this.popup != null) {
			RenderSystem.pushMatrix();
			RenderSystem.translatef(this.guiLeft, this.guiTop, 1000F);
			RenderSystem.color4f(1.0F, 1.0F, 1.0F, 1.0F);
			RenderHelper.disableStandardItemLighting();

			this.fillGradient(matrixStack, 0, 0, BACKGROUND_IMAGE_WIDTH, BACKGROUND_IMAGE_HEIGHT,
					new Color(16, 16, 16, 192).getRGB(), new Color(16, 16, 16, 208).getRGB());
			this.popup.draw(matrixStack, this.minecraft,
					this.popup.getRelXPosition(), this.popup.getRelYPosition(),
					mouseX - this.popup.getRelXPosition() - this.guiLeft,
					mouseY - this.popup.getRelYPosition() - this.guiTop);

			RenderSystem.popMatrix();
		}
	}

	@Override
	public void drawItemStack(ItemStack stack, int x, int y, String altText) {
		RenderSystem.pushMatrix();
		RenderSystem.translatef(0.0F, 0.0F, 32.0F);
		this.setBlitOffset(600);
		this.itemRenderer.zLevel = 200.0F;
		FontRenderer font = stack.getItem().getFontRenderer(stack);
		if(font == null) {
			font = this.font;
		}
		this.itemRenderer.renderItemAndEffectIntoGUI(stack, x, y);
		this.itemRenderer.renderItemOverlayIntoGUI(font, stack, x, y - (this.draggedStack.isEmpty() ? 0 : 8), altText);
		this.renderCustomItemStackOverlay(stack, x, y);
		this.setBlitOffset(0);
		this.itemRenderer.zLevel = 0.0F;
		RenderSystem.popMatrix();
	}

	private void renderCustomItemStackOverlay(ItemStack stack, int x, int y) {
		if(stack == null) {
			return;
		}
		String str;
		Item item = stack.getItem();
		if(item instanceof ItemNumber) {
			ItemNumber itemNumber = (ItemNumber) stack.getItem();
			int number = itemNumber.getNumber(stack);
			if(number == -1) {
				number = ItemNumber.DEFAULT_NUMBER;
			}
			str = (number >= 100 ? "..." : Integer.toString(number));
		} else if(item instanceof ItemVariable) {
			ItemVariable itemVariable = (ItemVariable) item;
			String name = itemVariable.getName(stack);
			str = (name.length() <= 2 ? name : (name.substring(0, 1) + "..."));
		} else {
			return;
		}

		MatrixStack matrixstack = new MatrixStack();
		matrixstack.translate(0, 0, this.itemRenderer.zLevel + 300f);
		float slotSize = 16;
		float fontHeight = 7;
		this.font.drawString(matrixstack, str, x + (slotSize - this.font.getStringWidth(str)) / 2f,
				y + (slotSize - fontHeight) / 2f, new Color(0, 0, 0).getRGB());
		RenderSystem.enableDepthTest();
	}

	/**
	 * This method is run at everyframe to draw the background of the GUI. We use this as an entrance for changing
	 * the screen as we have access to the TurtleContainer here
	 *
	 * @param partialTicks partial tick if your frame rate is higher
	 * @param mouseX XLocation of the Mouse
	 * @param mouseY YLocation of the Mouse
	 */
	@Override
	protected void drawGuiContainerBackgroundLayer(
			MatrixStack matrixStack, float partialTicks, int mouseX, int mouseY) {
		RenderSystem.color4f(1.0f, 1.0f, 1.0f, 1.0f);
		this.minecraft.getTextureManager().bindTexture(SCGlobal.BACKGROUND_TURTLE_TEXTURE);
		this.blit(matrixStack, this.guiLeft, this.guiTop, 0, 0, this.xSize, this.ySize);
		this.widgetManager.drawGuiContainerBackgroundLayer(matrixStack, mouseX, mouseY);
	}

	/**
	 * This method writes the images on the Foreground, This used to draw the text on of the TerminalCodeWriter.
	 * THis draws the written code as Lua code.
	 *
	 * @param mouseX vertical position of the mouse
	 * @param mouseY horizontal position of the mouse
	 */
	@Override
	protected void drawGuiContainerForegroundLayer(MatrixStack matrixStack, int mouseX, int mouseY) {

		// Draw the GUI foreground using absolute coordinates.
		RenderSystem.pushMatrix();
		RenderSystem.translatef(-this.guiLeft, -this.guiTop, 0.0F);
		this.widgetManager.drawGuiContainerForegroundLayer(matrixStack, mouseX, mouseY);
		RenderSystem.popMatrix();

		// Draw the name of the of the screen at the top of the screen.
		String screenName = this.widgetManager.getScreen().getScreenName();
		this.font.drawText(matrixStack, new StringTextComponent(screenName),
				X_OFFSET_LEFT, GUI_SCREEN_NAME_Y_OFFSET, SCGlobal.GUI_COLOR_DARK_GREY);

		// Draw coding inventory compile error overlays.
		this.renderCompileErrorOverlays(matrixStack);
	}

	/**
	 * Renders compile error overlays for compile errors in the coding inventory.
	 * @param matrixStack
	 */
	private void renderCompileErrorOverlays(MatrixStack matrixStack) {
		
		// Return if the coding inventory is not being rendered.
		if(this.widgetManager.getScreen() != TurtleInventoryScreen.PROGRAMMING_SCREEN) {
			return;
		}
		
		// Return if there are no compile errors.
		CompileResult compileResult = this.tileTurtleSC.getInventoryManager().getCodingInventory().getCompileResult();
		if(compileResult == null || compileResult.getCompileExceptions().isEmpty()) {
			return;
		}
		
		// Create a set of slots involved in compile errors.
		// A set is used here to prevent rendering duplicate overlays for slots that are involved in multiple errors.
		List<CompileException> compileExceptions = compileResult.getCompileExceptions();
		Set<Slot> errorSlots = new HashSet<>();
		for(CompileException compileEx : compileExceptions) {
			if(compileEx.getCodeStartIndex() >= 0 && compileEx.getCodeEndIndex() >= 0) {
				int slotEndIndex = (compileEx.getCodeEndIndex() < GuiConstants.CODING_INVENTORY_TOTAL_SLOTS
						? compileEx.getCodeEndIndex() : GuiConstants.CODING_INVENTORY_TOTAL_SLOTS - 1);
				for(int slotInd = compileEx.getCodeStartIndex(); slotInd <= slotEndIndex; slotInd++) {
					Slot slot = this.container.inventorySlots.get(
							GuiConstants.CODING_INVENTORY_SLOT_INDEX_OFFSET + slotInd);
					errorSlots.add(slot);
				}
			}
		}
		
		// Render red slot overlays.
		for(Slot slot : errorSlots) {
			RenderSystem.pushMatrix();
			RenderSystem.translatef(slot.xPos, slot.yPos, 0f);
			RenderSystem.disableDepthTest();
			RenderSystem.colorMask(true, true, true, false);

			int colorInt = new Color(255, 0, 0, 160).getRGB();
			fill(matrixStack, 0, 0, SLOT_SIZE - 2, SLOT_SIZE - 2, colorInt);

			RenderSystem.colorMask(true, true, true, true);
			RenderSystem.enableDepthTest();
			RenderSystem.popMatrix();
		}
	}

	@Override
	protected void renderHoveredTooltip(MatrixStack matrixStack, int mouseX, int mouseY) {

		// Return if the player is holding an item stack or if no item stack is hovered.
		if(!this.minecraft.player.inventory.getItemStack().isEmpty() || this.hoveredSlot == null) {
			return;
		}

		// Render custom error tooltips if there are compile errors on this code slot if it's in the coding inventory.
		if(this.hoveredSlot instanceof CodingSlot) {
			CompileResult compileResult =
					this.tileTurtleSC.getInventoryManager().getCodingInventory().getCompileResult();
			if(compileResult != null && !compileResult.getCompileExceptions().isEmpty()) {

				// List all exceptions on the currently hovered slot.
				List<CompileException> hoveredCompileExceptions = new ArrayList<>();
				for(CompileException compileEx : compileResult.getCompileExceptions()) {
					if(compileEx.getCodeStartIndex() <= this.hoveredSlot.getSlotIndex()
							&& compileEx.getCodeEndIndex() >= this.hoveredSlot.getSlotIndex()) {
						hoveredCompileExceptions.add(compileEx);
					}
				}

				// Render hovered slot errors tooltip.
				if(!hoveredCompileExceptions.isEmpty()) {
					List<IFormattableTextComponent> textCompList = new ArrayList<>();
					for(CompileException compileEx : hoveredCompileExceptions) {
						textCompList.add(compileEx.getTextComponent().mergeStyle(TextFormatting.DARK_RED));
					}
					RenderSystem.color4f(1.0f, 1.0f, 1.0f, 1.0f);
					this.renderWrappedToolTip(matrixStack, textCompList, mouseX, mouseY, this.font);
					return;
				}
			}
		}

		// Render custom button slot tooltip.
		if(this.hoveredSlot instanceof ButtonSlot) {
			ItemStack hoveredItemStack = this.hoveredSlot.getStack();

			// Render no popup when the button slot does not contain a GUI button item.
			if(hoveredItemStack.isEmpty() || !(hoveredItemStack.getItem() instanceof ItemGuiButton)) {
				return;
			}

			// Render custom button slot tooltip.
			FontRenderer font = hoveredItemStack.getItem().getFontRenderer(hoveredItemStack);
			List<IFormattableTextComponent> hoverTextComponents = new ArrayList<>();
			hoverTextComponents.add(
					new StringTextComponent("").appendSibling(hoveredItemStack.getDisplayName())
					.mergeStyle(TextFormatting.DARK_GREEN));
			hoverTextComponents.add(
					new TranslationTextComponent(hoveredItemStack.getTranslationKey() + ".hoverinfo")
					.mergeStyle(TextFormatting.GOLD));
			this.drawHoveringTextWithImage(hoveredItemStack, matrixStack, hoverTextComponents,
					new ResourceLocation(SCGlobal.MOD_ID, "textures/items/hoverinfo/"
									+ hoveredItemStack.getItem().getRegistryName().getPath() + ".png"),
					mouseX, mouseY, this.width, this.height, -1, GuiUtils.DEFAULT_BACKGROUND_COLOR,
					GuiUtils.DEFAULT_BORDER_COLOR_START, GuiUtils.DEFAULT_BORDER_COLOR_END,
					(font == null ? this.font : font));
			return;
		}

		// Render custom programming item tooltip for programming items.
		ItemStack hoveredItemStack = this.hoveredSlot.getStack();
		if(!hoveredItemStack.isEmpty()) {
			if(hoveredItemStack.getItem() instanceof ItemProgrammingIcon) {

				// Render custom programming item tooltip.
				FontRenderer font = hoveredItemStack.getItem().getFontRenderer(hoveredItemStack);
				List<IFormattableTextComponent> hoverTextComponents = new ArrayList<>();
				hoverTextComponents.add(
						new StringTextComponent("").appendSibling(hoveredItemStack.getDisplayName())
						.mergeStyle(TextFormatting.DARK_GREEN));
				hoverTextComponents.add(
						new TranslationTextComponent(hoveredItemStack.getTranslationKey() + ".hoverinfo")
						.mergeStyle(TextFormatting.GOLD));
				if(this.hoveredSlot instanceof ProgrammingSlot) {
					hoverTextComponents.add(new TranslationTextComponent(
							"gui.computercraftsc.shift_click_to_place_item").mergeStyle(TextFormatting.GRAY));
				} else if(this.hoveredSlot instanceof CodingSlot) {
					hoverTextComponents.add(new TranslationTextComponent(
							"gui.computercraftsc.shift_click_to_remove_item").mergeStyle(TextFormatting.GRAY));
				}
				this.drawHoveringTextWithImage(hoveredItemStack, matrixStack, hoverTextComponents,
						new ResourceLocation(SCGlobal.MOD_ID, "textures/items/hoverinfo/"
										+ hoveredItemStack.getItem().getRegistryName().getPath() + ".png"),
						mouseX, mouseY, this.width, this.height, -1, GuiUtils.DEFAULT_BACKGROUND_COLOR,
						GuiUtils.DEFAULT_BORDER_COLOR_START, GuiUtils.DEFAULT_BORDER_COLOR_END,
						(font == null ? this.font : font));
			} else {

				// Render default tooltip.
				this.renderTooltip(matrixStack, hoveredItemStack, mouseX, mouseY);
			}
		}
	}

	private void drawHoveringTextWithImage(@Nonnull final ItemStack stack, MatrixStack mStack,
			List<? extends ITextProperties> textLines, ResourceLocation imageLocation, int mouseX, int mouseY,
			int screenWidth, int screenHeight, int maxTextWidth,
			int backgroundColor, int borderColorStart, int borderColorEnd, FontRenderer font) {
		if(textLines.isEmpty()) {
			return;
		}

		RenderSystem.disableRescaleNormal();
		RenderSystem.disableDepthTest();
		int tooltipTextWidth = 0;
		for(ITextProperties textLine : textLines) {
			int textLineWidth = font.getStringPropertyWidth(textLine);
			if(textLineWidth > tooltipTextWidth) {
				tooltipTextWidth = textLineWidth;
			}
		}

		boolean needsWrap = false;
		int titleLinesCount = 1;
		int tooltipX = mouseX + 12;
		if(tooltipX + tooltipTextWidth + 4 > screenWidth) {
			tooltipX = mouseX - 16 - tooltipTextWidth;
			if(tooltipX < 4) { // If the tooltip doesn't fit on the screen.

				if(mouseX > screenWidth / 2) {
					tooltipTextWidth = mouseX - 12 - 8;
				} else {
					tooltipTextWidth = screenWidth - 16 - mouseX;
				}
				needsWrap = true;
			}
		}

		if(maxTextWidth > 0 && tooltipTextWidth > maxTextWidth) {
			tooltipTextWidth = maxTextWidth;
			needsWrap = true;
		}

		if(needsWrap) {
			int wrappedTooltipWidth = 0;
			List<ITextProperties> wrappedTextLines = new ArrayList<>();
			for(int i = 0; i < textLines.size(); i++) {
				ITextProperties textLine = textLines.get(i);
				List<ITextProperties> wrappedLine = font.getCharacterManager()
						.func_238362_b_(textLine, tooltipTextWidth, Style.EMPTY);
				if(i == 0) {
					titleLinesCount = wrappedLine.size();
				}

				for(ITextProperties line : wrappedLine) {
					int lineWidth = font.getStringPropertyWidth(line);
					if(lineWidth > wrappedTooltipWidth) {
						wrappedTooltipWidth = lineWidth;
					}
					wrappedTextLines.add(line);
				}
			}
			tooltipTextWidth = wrappedTooltipWidth;
			textLines = wrappedTextLines;

			if(mouseX > screenWidth / 2) {
				tooltipX = mouseX - 16 - tooltipTextWidth;
			} else {
				tooltipX = mouseX + 12;
			}
		}

		int tooltipY = mouseY - 12;
		int tooltipHeight = 8;

		if(textLines.size() > 1) {
			tooltipHeight += (textLines.size() - 1) * 10;
			if(textLines.size() > titleLinesCount)
			tooltipHeight += 2; // Gap between title lines and next lines.
		}

		if(tooltipY < 4) {
			tooltipY = 4;
		} else if(tooltipY + tooltipHeight + 4 > screenHeight) {
			tooltipY = screenHeight - tooltipHeight - 4;
		}

		// Get image dimensions.
		int imageWidth = -1;
		int imageHeight = -1;
		if(this.minecraft.getResourceManager().hasResource(imageLocation)) {
			TextureData textureData = TextureData.getTextureData(this.minecraft.getResourceManager(), imageLocation);
			try {
				NativeImage nativeImage = textureData.getNativeImage();
				imageWidth = nativeImage.getWidth();
				imageHeight = nativeImage.getHeight();
			} catch (IOException e) {
				LOGGER.warn("Exception while loading hovering text image: " + imageLocation.toString(), e);
			}
		}

		// Enlarge tooltip to make space to include the image.
		final int imageTextSpacing = 4;
		final int imageBorderSpacing = 0; // Zero extra spacing has the same spacing as the text.
		if(imageHeight > 0) {
			tooltipHeight += imageTextSpacing + imageHeight + imageBorderSpacing;
			if(imageWidth + 2 * imageBorderSpacing > tooltipTextWidth) {
				tooltipTextWidth = imageWidth + 2 * imageBorderSpacing;
			}
		}

		// Draw tooltip background.
		final int zLevel = 400;
		mStack.push();
		Matrix4f mat = mStack.getLast().getMatrix();

		// Top black line, 1 pixel height.
		GuiUtils.drawGradientRect(mat, zLevel,
				tooltipX - 3,
				tooltipY - 4,
				tooltipX + tooltipTextWidth + 3,
				tooltipY - 3,
				backgroundColor, backgroundColor);

		// Bottom black line, 1 pixel height.
		GuiUtils.drawGradientRect(mat, zLevel,
				tooltipX - 3,
				tooltipY + tooltipHeight + 3,
				tooltipX + tooltipTextWidth + 3,
				tooltipY + tooltipHeight + 4,
				backgroundColor, backgroundColor);

		// Black background.
		GuiUtils.drawGradientRect(mat, zLevel,
				tooltipX - 3,
				tooltipY - 3,
				tooltipX + tooltipTextWidth + 3,
				tooltipY + tooltipHeight + 3,
				backgroundColor, backgroundColor);

		// Left black line, 1 pixel width.
		GuiUtils.drawGradientRect(mat, zLevel,
				tooltipX - 4,
				tooltipY - 3,
				tooltipX - 3,
				tooltipY + tooltipHeight + 3,
				backgroundColor, backgroundColor);

		// Right black line, 1 pixel width.
		GuiUtils.drawGradientRect(mat, zLevel,
				tooltipX + tooltipTextWidth + 3,
				tooltipY - 3,
				tooltipX + tooltipTextWidth + 4,
				tooltipY + tooltipHeight + 3,
				backgroundColor, backgroundColor);

		// Left purple line, 1 pixel width.
		GuiUtils.drawGradientRect(mat, zLevel,
				tooltipX - 3,
				tooltipY - 3 + 1,
				tooltipX - 3 + 1,
				tooltipY + tooltipHeight + 3 - 1,
				borderColorStart, borderColorEnd);

		// Right purple line, 1 pixel width.
		GuiUtils.drawGradientRect(mat, zLevel,
				tooltipX + tooltipTextWidth + 2,
				tooltipY - 3 + 1,
				tooltipX + tooltipTextWidth + 3,
				tooltipY + tooltipHeight + 3 - 1,
				borderColorStart, borderColorEnd);

		// Top purple line, 1 pixel height.
		GuiUtils.drawGradientRect(mat, zLevel,
				tooltipX - 3,
				tooltipY - 3,
				tooltipX + tooltipTextWidth + 3,
				tooltipY - 3 + 1,
				borderColorStart, borderColorStart);

		// Bottom purple line, 1 pixel height.
		GuiUtils.drawGradientRect(mat, zLevel,
				tooltipX - 3,
				tooltipY + tooltipHeight + 2,
				tooltipX + tooltipTextWidth + 3,
				tooltipY + tooltipHeight + 3,
				borderColorEnd, borderColorEnd);

		IRenderTypeBuffer.Impl renderType = IRenderTypeBuffer.getImpl(Tessellator.getInstance().getBuffer());
		mStack.translate(0.0D, 0.0D, zLevel);

		// Draw text.
		int textY = tooltipY;
		for(int lineNumber = 0; lineNumber < textLines.size(); lineNumber++) {
			ITextProperties line = textLines.get(lineNumber);
			if(line != null) {
				font.drawEntityText(LanguageMap.getInstance().func_241870_a(line),
						(float) tooltipX, (float) textY, -1, true, mat, renderType, false, 0, 15728880);
			}
			if(lineNumber + 1 == titleLinesCount) {
				textY += 2;
			}
			textY += 10;
		}
		renderType.finish();

		// Draw image if available.
		if(imageWidth > 0) {
			this.minecraft.getTextureManager().bindTexture(imageLocation);
			AbstractGui.blit(mStack,
					tooltipX + imageBorderSpacing, tooltipY + tooltipHeight - imageBorderSpacing - imageHeight,
					0, 0, imageWidth, imageHeight, imageWidth, imageHeight);
		}
		mStack.pop();

		RenderSystem.enableDepthTest();
		RenderSystem.enableRescaleNormal();
	}

	@Override
	public void showPopup(PopupType type, int popupX, int popupY, int slot) {
		switch(type) {
			case NUMBER: {
				this.popup = new WidgetTextEntryPopup(popupX, popupY, "",
						new WidgetTextEntryPopup.ITextEntryListener() {

					@Override
					public void onCancel(WidgetTextEntryPopup widget) {
						TurtleGuiContainerSC.this.closePopup();
					}

					@Override
					public void onConfirmation(WidgetTextEntryPopup widget, String input) {

						// Replace the item stack in the inventory if it still is an ItemNumber stack.
						ItemStack itemStack = TurtleGuiContainerSC.this.tileTurtleSC.getStackInSlot(slot);
						if(itemStack != null && itemStack.getItem() instanceof ItemNumber) {
							ItemNumber itemNumber = (ItemNumber) itemStack.getItem();
							TurtleGuiContainerSC.this.setProgramStackThroughServer(
									slot, itemNumber.create(Integer.parseInt(input)));
						} else {
							LOGGER.warn("Cannot set new item. Original item has changed after the popup has opened.");
						}

						// Close the popup.
						TurtleGuiContainerSC.this.closePopup();
					}
				}, TextType.NUMBER);
				break;
			}
			case STRING: {
				this.popup = new WidgetTextEntryPopup(popupX, popupY, "",
						new WidgetTextEntryPopup.ITextEntryListener() {

					@Override
					public void onCancel(WidgetTextEntryPopup widget) {
						TurtleGuiContainerSC.this.closePopup();
					}

					@Override
					public void onConfirmation(WidgetTextEntryPopup widget, String input) {

						// Replace the item stack in the inventory if it still is an ItemString/ItemComment stack.
						ItemStack itemStack = TurtleGuiContainerSC.this.tileTurtleSC.getStackInSlot(slot);
						if(itemStack != null) {
							if(itemStack.getItem() instanceof ItemString) {
								ItemString itemString = (ItemString) itemStack.getItem();
								TurtleGuiContainerSC.this.setProgramStackThroughServer(slot, itemString.create(input));
							} else if(itemStack.getItem() instanceof ItemComment) {
								ItemComment itemComment = (ItemComment) itemStack.getItem();
								TurtleGuiContainerSC.this.setProgramStackThroughServer(slot, itemComment.create(input));
							}
						} else {
							LOGGER.warn("Cannot set new item. Original item has changed after the popup has opened.");
						}

						// Close the popup.
						TurtleGuiContainerSC.this.closePopup();
					}
				}, TextType.STRING);
				break;
			}
			case VARIABLE: {
				this.popup = new WidgetTextEntryPopup(popupX, popupY, "",
						new WidgetTextEntryPopup.ITextEntryListener() {

					@Override
					public void onCancel(WidgetTextEntryPopup widget) {
						TurtleGuiContainerSC.this.closePopup();
					}

					@Override
					public void onConfirmation(WidgetTextEntryPopup widget, String input) {

						// Replace the item stack in the inventory if it still is an ItemVariable stack.
						ItemStack itemStack = TurtleGuiContainerSC.this.tileTurtleSC.getStackInSlot(slot);
						if(itemStack != null && itemStack.getItem() instanceof ItemVariable) {
							ItemVariable itemVariable = (ItemVariable) itemStack.getItem();
							TurtleGuiContainerSC.this.setProgramStackThroughServer(slot, itemVariable.create(input));
						} else {
							LOGGER.warn("Cannot set new item. Original item has changed after the popup has opened.");
						}

						// Close the popup.
						TurtleGuiContainerSC.this.closePopup();
					}
				}, TextType.SYMBOL);
				break;
			}
			case BLOCK_NAME: {
				this.popup = new WidgetBlockSelectionPopup(
						popupX, popupY, ItemBlockName.DEFAULT_BLOCK, new IBlockSelectionListener() {

					@Override
					public void onCancel(WidgetBlockSelectionPopup widget) {
						TurtleGuiContainerSC.this.closePopup();
					}

					@Override
					public void onConfirmation(WidgetBlockSelectionPopup widget, Block input) {

						// Replace the item stack in the inventory if it still is an ItemBlockName stack.
						ItemStack itemStack = TurtleGuiContainerSC.this.tileTurtleSC.getStackInSlot(slot);
						if(itemStack.getItem() instanceof ItemBlockName) {
							ItemBlockName itemBlockName = (ItemBlockName) itemStack.getItem();
							TurtleGuiContainerSC.this.setProgramStackThroughServer(slot, itemBlockName.create(input));
						} else {
							LOGGER.warn("Cannot set new item. Original item has changed after the popup has opened.");
						}

						// Close the popup.
						TurtleGuiContainerSC.this.closePopup();
					}
				});
				break;
			}
			case ITEM_NAME: {
				this.popup = new WidgetItemSelectionPopup(
						popupX, popupY, ItemItemName.DEFAULT_ITEM, new IItemSelectionListener() {

					@Override
					public void onCancel(WidgetItemSelectionPopup widget) {
						TurtleGuiContainerSC.this.closePopup();
					}

					@Override
					public void onConfirmation(WidgetItemSelectionPopup widget, Item input) {

						// Replace the item stack in the inventory if it still is an ItemItemName stack.
						ItemStack itemStack = TurtleGuiContainerSC.this.tileTurtleSC.getStackInSlot(slot);
						if(itemStack.getItem() instanceof ItemItemName) {
							ItemItemName itemItemName = (ItemItemName) itemStack.getItem();
							TurtleGuiContainerSC.this.setProgramStackThroughServer(slot, itemItemName.create(input));
						} else {
							LOGGER.warn("Cannot set new item. Original item has changed after the popup has opened.");
						}

						// Close the popup.
						TurtleGuiContainerSC.this.closePopup();
					}
				});
				break;
			}
			default: {
				throw new Error("Unsupported " + PopupType.class.getSimpleName() + ": " + type);
			}
		}
	}

	public void closePopup() {
		this.popup = null;
	}

	/**
	 * Sets the given {@link ItemStack} into the given slot through the server. This allows clients to create/alter
	 * items without being in creative mode.
	 * @param slot - The slot number.
	 * @param itemStack - The {@link ItemStack}.
	 */
	private void setProgramStackThroughServer(int slot, ItemStack itemStack) {
		NetworkManagerSC.sendPacketToServer(new SetCodingSlotStackPacket(slot, itemStack));
	}

	@Override
	public void tick() {

		// Tick popup if it exists.
		if(this.popup != null) {
			this.popup.tick();
		}

		super.tick();
	}

	@Override
	public boolean keyPressed(int keyCode, int scanCode, int modifiers) {

		// Pass the key press to the popup if it exists.
		if(this.popup != null && this.popup.keyPressed(keyCode, scanCode, modifiers)) {
			return true;
		}

		// Pass the key press to the active widget.
		if(this.widgetManager.getActiveWidgetContainer().keyPressed(keyCode, scanCode, modifiers)) {
			return true;
		}

		// Activate the run button when pressing enter.
		if(keyCode == KeyCodes.RETURN || keyCode == KeyCodes.NUM_RETURN) {

			// Send a run button mouse click to the server to activate the run button.
			Slot runButtonSlot = this.container.getSlotManager().getRunButtonSlot();
			this.handleMouseClick(runButtonSlot, runButtonSlot.slotNumber, 0, ClickType.PICKUP);
			return true;
		}

		// Pass the key press to the super implementation.
		return super.keyPressed(keyCode, scanCode, modifiers);
	}

	@Override
	public boolean charTyped(char ch, int modifiers) {

		// Pass the char typed to the popup if it exists.
		if(this.popup != null && this.popup.charTyped(ch, modifiers)) {
			return true;
		}

		// Pass the char typed to the active widget.
		if(this.widgetManager.getActiveWidgetContainer().charTyped(ch, modifiers)) {
			return true;
		}

		return super.charTyped(ch, modifiers);
	}

	/**
	 * Note that this gets called on mouse press instead of mouse click (press + release).
	 */
	@Override
	public boolean mouseClicked(double mouseX, double mouseY, int mouseButton) {

		// Pass the click to the popup if it exists.
		if(this.popup != null) {
			this.popup.mousePressed(
					(int) mouseX - this.popup.getAbsoluteXPosition() - this.guiLeft,
					(int) mouseY - this.popup.getAbsoluteYPosition() - this.guiTop, mouseButton);
			return true;
		}

		// Pass the mouse press to the active widget.
		WidgetContainer activeContainer = this.widgetManager.getActiveWidgetContainer();
		if(activeContainer.mousePressed(
				(int) mouseX - activeContainer.getAbsoluteXPosition(),
				(int) mouseY - activeContainer.getAbsoluteYPosition(), mouseButton)) {
			return true;
		}

		// Pass the click to the other GUI components.
		boolean handled = super.mouseClicked(mouseX, mouseY, mouseButton);

		// Reset double click flag for programming and coding slots to disable double clicks for them.
		if(this.lastClickSlot instanceof CodingSlot || this.lastClickSlot instanceof ProgrammingSlot) {
			this.doubleClick = false;
		}

		// Return the super call result.
		return handled;
	}

	@Override
	public boolean mouseReleased(double mouseX, double mouseY, int mouseButton) {

		// Pass the release to the popup if it exists.
		if(this.popup != null) {
			this.popup.mouseReleased(
					(int) mouseX - this.popup.getAbsoluteXPosition() - this.guiLeft,
					(int) mouseY - this.popup.getAbsoluteYPosition() - this.guiTop, mouseButton);
			return true;
		}

		// Pass the mouse release to the active widget.
		WidgetContainer activeContainer = this.widgetManager.getActiveWidgetContainer();
		if(activeContainer.mouseReleased(
				(int) mouseX - activeContainer.getAbsoluteXPosition(),
				(int) mouseY - activeContainer.getAbsoluteYPosition(), mouseButton)) {
			return true;
		}

		// Pass the release to the other GUI components.
		return super.mouseReleased(mouseX, mouseY, mouseButton);
	}

	@Override
	public void mouseMoved(double mouseX, double mouseY) {
		super.mouseMoved(mouseX, mouseY);
		if(this.popup != null) {
			this.popup.mouseMoved(
					(int) mouseX - this.popup.getAbsoluteXPosition() - this.guiLeft,
					(int) mouseY - this.popup.getAbsoluteYPosition() - this.guiTop);
		}

		// Pass the mouse move to the active widget.
		WidgetContainer activeContainer = this.widgetManager.getActiveWidgetContainer();
		activeContainer.mouseMoved(
				(int) mouseX - activeContainer.getAbsoluteXPosition(),
				(int) mouseY - activeContainer.getAbsoluteYPosition());
	}

	@Override
	public boolean mouseScrolled(double mouseX, double mouseY, double scrollDelta) {

		// Pass the scroll to the popup if it exists.
		if(this.popup != null) {
			this.popup.mouseScrolled(
					(int) mouseX - this.popup.getAbsoluteXPosition() - this.guiLeft,
					(int) mouseY - this.popup.getAbsoluteYPosition() - this.guiTop, scrollDelta);
			return true;
		}

		// Pass the mouse press to the active widget.
		WidgetContainer activeContainer = this.widgetManager.getActiveWidgetContainer();
		if(activeContainer.mouseScrolled(
				(int) mouseX - activeContainer.getAbsoluteXPosition(),
				(int) mouseY - activeContainer.getAbsoluteYPosition(), scrollDelta)) {
			return true;
		}

		// Pass the scroll to the other GUI components.
		return super.mouseScrolled(mouseX, mouseY, scrollDelta);
	}

	@Override
	public void setGuiScreen(TurtleInventoryScreen screen) {
		this.widgetManager.setGuiScreen(screen);
	}

	@Override
	public Container getContainerObj() {
		return this.container;
	}

	@Override
	public TurtleContainerSC getContainer() {
		return this.container;
	}
}
