package computercraftsc.client.gui.container.slotmanager;

import computercraftsc.client.gui.inventories.Inventory;
import net.minecraft.inventory.container.Slot;

import java.util.List;

import static computercraftsc.client.gui.GuiConstants.SLOT_SIZE;
import static computercraftsc.client.gui.GuiConstants.INVISIBLE_LOCATION;


/**
 * This Class manages the slots bound to their inventory.
 * This way you can link the slots by inventory, and place them somewhere the Player cant see, or reset them
 * where you want to.
 */
public class InventorySlotManager {

	protected final int xOffset;
	protected final int yOffset;
	protected final List<Slot> slots;
	protected final Inventory inventory;
	protected boolean isVisible = true;
	private int slotRowScrollOffset = 0;

	/**
	 *
	 * @param slots All the slots paired to this CustomAbstractInventory
	 * @param inventory The CustomAbstractInventory is mainly used for it knowledge of its Columns and Rows
	 * @param xOffset The XCoordinate where the first slot should be placed
	 * @param yOffset The YCoordiante where the first slot should be placed
	 */
	public InventorySlotManager(List<Slot> slots, Inventory inventory, int xOffset, int yOffset) {
		this.slots = slots;
		this.inventory = inventory;
		this.xOffset = xOffset + 1; // +1 to center the slot in its container
		this.yOffset = yOffset + 1; // +1 to center the slot in its container
	}

	/**
	 * Gets the slot at the given inventory index.
	 * @param inventoryIndex - The index of the slot in the inventory.
	 * @return The {@link Slot} at the given index.
	 */
	public Slot getSlot(int inventoryIndex) {
		return this.slots.get(inventoryIndex);
	}

	/**
	 * Use this function to place the slots where The player can see.
	 * Keep in mind this always loads the slots in a Grid as Defined by its columns and Rows
	 */
	public void setSlotsVisible(boolean visible) {
		if (this.isVisible != visible) {
			this.isVisible = visible;
			this.makeAllSlotsInvisible();
			this.scrollToRow(0);
		}
	}

	/**
	 * THis method is used to places the slots in a row of the grid. This method is to make the setSlotsVisible() method more readable.
	 * Also removes this some duplicate code that would be written for classes that extend this class.
	 * @param firstOffset The Index of the first slot to be placed
	 * @param totalColumns The amount of Collumns in this row
	 * @param xOffset X Coordinate of the first slot
	 * @param yOffset Y Coordinate of the first slot
	 * @return The Index at the end of this method.
	 */
	protected int placeSlotsInRow(int firstOffset, int totalColumns, int xOffset, int yOffset) {
		int newOffset = firstOffset + totalColumns;
		for (int index = firstOffset; index < newOffset; index++) {
			Slot currentSlot = this.slots.get(index);
			currentSlot.xPos = xOffset;
			currentSlot.yPos = yOffset;
			xOffset += SLOT_SIZE;
		}
		return newOffset;
	}

	public void scrollToRow(int topRow) {
		if (!this.isVisible) {
			return;
		}

		this.makeAllSlotsInvisible();

		this.slotRowScrollOffset = topRow;
		int slotInd = topRow * this.inventory.getColumns();
		int yOffset = this.yOffset;
		for (int rows = 0; rows < this.inventory.getVisibleRows() && slotInd < this.inventory.getSizeInventory(); rows++) {
			slotInd = this.placeSlotsInRow(slotInd, this.inventory.getColumns(), this.xOffset, yOffset);
			yOffset += SLOT_SIZE;
		}
	}

	/**
	 * Scrolls the least amount of rows to make the slot at the given index visible in the view.
	 * Does nothing if the slot is already visible.
	 * @param slotInd - The slot index.
	 */
	public void scrollSlotToView(int slotInd) {
		if(slotInd < 0 || slotInd > this.slots.size()) {
			return;
		}
		int slotStartInd = this.slotRowScrollOffset * this.inventory.getColumns();
		int slotEndInd = slotStartInd + this.inventory.getVisibleRows() * this.inventory.getColumns() - 1;
		if(slotInd >= slotStartInd && slotInd <= slotEndInd) {
			return;
		}
		if(slotInd < slotStartInd) {

			// Scroll up.
			int slotRow = slotInd / this.inventory.getColumns();
			this.scrollToRow(slotRow);
		} else if(slotInd > slotEndInd) {

			// Scroll down.
			int slotRow = slotInd / this.inventory.getColumns();
			int topRow = slotRow - this.inventory.getVisibleRows() + 1;
			this.scrollToRow(topRow);
		}
	}

	/**
	 * Gets the index of the top visible slot row.
	 * @return The slot row scroll offset.
	 */
	public int getSlotRowScrollOffset() {
		return this.slotRowScrollOffset;
	}

	/**
	 * moves all slots to invisible locations
	 */
	private void makeAllSlotsInvisible() {
		for (Slot slot : this.slots) {
			slot.xPos = INVISIBLE_LOCATION;
			slot.yPos = INVISIBLE_LOCATION;
		}
	}
}
