package computercraftsc.client.gui.container.slotmanager;

import computercraftsc.client.gui.inventories.InventoryManager;
import computercraftsc.client.gui.TurtleInventoryScreen;
import computercraftsc.client.gui.container.GuiScreenHandler;
import computercraftsc.client.gui.container.TurtleContainerSC;
import computercraftsc.client.gui.container.slot.ButtonSlot;
import computercraftsc.client.gui.container.slot.GarbageButtonSlot;
import computercraftsc.client.gui.container.slot.RunButtonSlot;
import computercraftsc.client.gui.inventories.CodingInventory;
import computercraftsc.client.gui.inventories.Inventory;
import computercraftsc.client.gui.inventories.CCSCPlayerInventory;
import computercraftsc.client.gui.inventories.ProgrammingInventory;
import computercraftsc.client.gui.inventories.TurtleInventory;
import computercraftsc.shared.turtle.block.TileTurtleSC;
import net.minecraft.entity.player.PlayerEntity;
import net.minecraft.inventory.container.ClickType;
import net.minecraft.inventory.container.Slot;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static computercraftsc.client.gui.GuiConstants.ACTION_BUTTON_LENGTH;
import static computercraftsc.client.gui.GuiConstants.ACTION_BUTTON_X_OFFSET;
import static computercraftsc.client.gui.GuiConstants.ACTION_BUTTON_Y_OFFSET;
import static computercraftsc.client.gui.GuiConstants.CLOSE_BUTTON_X_OFFSET;
import static computercraftsc.client.gui.GuiConstants.CLOSE_BUTTON_Y_OFFSET;
import static computercraftsc.client.gui.GuiConstants.GARBAGE_BUTTON_X_OFFSET;
import static computercraftsc.client.gui.GuiConstants.GARBAGE_BUTTON_Y_OFFSET;
import static computercraftsc.client.gui.GuiConstants.INVISIBLE_LOCATION;
import static computercraftsc.client.gui.GuiConstants.RUN_BUTTON_X_OFFSET;
import static computercraftsc.client.gui.GuiConstants.RUN_BUTTON_Y_OFFSET;
import static computercraftsc.client.gui.GuiConstants.TURTLE_INVENTORY_X_OFFSET;
import static computercraftsc.client.gui.GuiConstants.X_OFFSET_LEFT;
import static computercraftsc.client.gui.GuiConstants.Y_OFFSET_TOP;
import static computercraftsc.client.gui.GuiConstants.PROGRAMMING_INVENTORY_X_OFFSET;
import static computercraftsc.client.gui.TurtleInventoryScreen.INVENTORY_SCREEN;
import static computercraftsc.client.gui.TurtleInventoryScreen.PROGRAMMING_SCREEN;
import static computercraftsc.client.gui.TurtleInventoryScreen.TERMINAL_SCREEN;


/**
 * This class is responsible for placing the slots on the screen. When you change to a different screen, slots that you
 * do not want to see are placed off screen, and slots you do want to see are placed at the designated locations
 */
public class SlotManager {

	private final TurtleContainerSC turtleContainerSC;
	private final TileTurtleSC tileTurtleSC;
	private final Map<TurtleInventoryScreen, List<InventorySlotManager>> slotInventoryManagers = new HashMap<>();
	private InventorySlotManager codingInventorySlotManager;
	private InventorySlotManager programmingInventorySlotManager;
	private GuiScreenHandler guiScreenHandler = null;
	private Slot garbageButtonSlot;
	private Slot runButtonSlot;

	/**
	 * The constructor takes
	 *
	 * @param turtleContainerSC The container is passed to tell it where to place the slots
	 * @param inventoryManager  The Inventory inventoryManager is passed to tell it the other of the inventories and their sltos
	 * @param tileTurtleSC      The TileEntity is passed to delegate the Action Button Slots to their actions
	 */
	public SlotManager(TurtleContainerSC turtleContainerSC, InventoryManager inventoryManager, TileTurtleSC tileTurtleSC) {
		this.turtleContainerSC = turtleContainerSC;
		this.tileTurtleSC = tileTurtleSC;

		this.setupSlotInventoryManagers(inventoryManager);
		this.setupActionButtons(inventoryManager);
	}

	private void setupSlotInventoryManagers(InventoryManager inventoryManager) {
		// setup inventoryScreen SlotManagers
		// IMPORTANT THAT THIS HAPPENS BEFORE PROGRAMMING SCREEN! for some reason, changing it will make some slots in
		// turtle and player inventory unusable (even more when in configMode)...
		ArrayList<InventorySlotManager> inventorySlotManagers = new ArrayList<>();

		TurtleInventory turtleInventory = inventoryManager.getTurtleInventory();
		List<Slot> turtleSlots = this.createAndAddSlotsForInventory(turtleInventory, turtleInventory.getMaxSlotsShown());
		inventorySlotManagers.add(new InventorySlotManager(turtleSlots, turtleInventory, TURTLE_INVENTORY_X_OFFSET, Y_OFFSET_TOP));

		CCSCPlayerInventory playerInventory = inventoryManager.getPlayerInventory();
		List<Slot> playerSlots = this.createAndAddSlotsForInventory(playerInventory, playerInventory.getMaxSlotsShown());
		inventorySlotManagers.add(new PlayerInventorySlotManager(playerSlots, playerInventory, X_OFFSET_LEFT, Y_OFFSET_TOP));

		// setup programmingScreen SlotManagers
		ArrayList<InventorySlotManager> programmingSlotManagers = new ArrayList<>();

		CodingInventory codingInventory = inventoryManager.getCodingInventory();
		List<Slot> codingSlots = this.createAndAddSlotsForInventory(codingInventory, codingInventory.getSizeInventory());
		this.codingInventorySlotManager = new InventorySlotManager(codingSlots, codingInventory, X_OFFSET_LEFT, Y_OFFSET_TOP);
		programmingSlotManagers.add(this.codingInventorySlotManager);

		ProgrammingInventory programmingInventory = inventoryManager.getProgrammingInventory();
		List<Slot> programmingSlots = this.createAndAddSlotsForInventory(programmingInventory, programmingInventory.getSizeInventory());
		this.programmingInventorySlotManager = new InventorySlotManager(programmingSlots, programmingInventory, PROGRAMMING_INVENTORY_X_OFFSET, Y_OFFSET_TOP);
		programmingSlotManagers.add(this.programmingInventorySlotManager);

		// setup terminalScreen SlotManagers
		ArrayList<InventorySlotManager> terminalSlotManagers = new ArrayList<>();

		// update this.slotInventoryManagers map with the SlotManagers
		this.slotInventoryManagers.put(PROGRAMMING_SCREEN, programmingSlotManagers);
		this.slotInventoryManagers.put(INVENTORY_SCREEN, inventorySlotManagers);
		this.slotInventoryManagers.put(TERMINAL_SCREEN, terminalSlotManagers);
	}

	/**
	 * This method places the Slots of the Inventory in the Container. This way the Container can store the items that
	 * are put in the the slots. An amount of slots is added equal to the amount of items to be stored in that inventory.
	 * The Slots, after they are placed in the inventory. Are returned to be used by a InventorySlotManager to handle
	 * their location on screen, based on the state the Gui is in.
	 * The Index is relative to the inventory - The initial location is irrelevant as it is moved base on its state
	 *
	 * @param inventory The {@link Inventory} backing the slot.
	 * @param size      Number of slots to be placed
	 * @return List of Slots set on the inventory.
	 */
	private List<Slot> createAndAddSlotsForInventory(Inventory inventory, int size) {
		ArrayList<Slot> listOfSlots = new ArrayList<>();
		for (int i = 0; i < size; i++) {
			Slot slot = inventory.createSlot(i, 0, 0);
			this.turtleContainerSC.addSlot(slot);
			listOfSlots.add(slot);
		}
		return listOfSlots;
	}

	/**
	 * THis method places the CloseButton and the ToggleScreenButton. These are created separately because these
	 * do not change on screen change. Their location are therefor hard coded.
	 * These are also added the the Managers Inventory to avoid any confusion with inventories that would be added in the future.
	 * @param inventoryManager inventoryManager to access buttons
	 */
	private void setupActionButtons(InventoryManager inventoryManager) {

		// Create close button slot. 
		Slot closeButtonSlot = new ButtonSlot(inventoryManager.getCloseButton(), CLOSE_BUTTON_X_OFFSET, CLOSE_BUTTON_Y_OFFSET) {
			@Override
			public void onSlotClicked(ClickType clickType, int clickedButton, PlayerEntity player) {
				if(clickType == ClickType.PICKUP) {
					player.closeScreen();
				}
			}
		};
		this.turtleContainerSC.addSlot(closeButtonSlot);

		// Create garbage button slot.
		this.garbageButtonSlot = new GarbageButtonSlot(this.tileTurtleSC,
				inventoryManager.getGarbageButton(), GARBAGE_BUTTON_X_OFFSET, GARBAGE_BUTTON_Y_OFFSET);
		this.turtleContainerSC.addSlot(this.garbageButtonSlot);

		// Create run button slot.
		this.runButtonSlot = new RunButtonSlot(this.tileTurtleSC,
				inventoryManager.getPlayButton(), RUN_BUTTON_X_OFFSET, RUN_BUTTON_Y_OFFSET);
		this.turtleContainerSC.addSlot(this.runButtonSlot);

		// Create screen tab slots which can be clicked to switch screens.
		int actionButtonXOffset = ACTION_BUTTON_X_OFFSET;
		Slot programmingSlot = new ButtonSlot(inventoryManager.getProgrammingScreenButton(), actionButtonXOffset, ACTION_BUTTON_Y_OFFSET) {
			@Override
			public void onSlotClicked(ClickType clickType, int clickedButton, PlayerEntity player) {
				if(SlotManager.this.guiScreenHandler != null
						&& clickType == ClickType.PICKUP && player.inventory.getItemStack().isEmpty()) {
					SlotManager.this.guiScreenHandler.setGuiScreen(PROGRAMMING_SCREEN);
				}
			}
		};
		this.turtleContainerSC.addSlot(programmingSlot);

		actionButtonXOffset += ACTION_BUTTON_LENGTH;
		Slot inventorySlot = new ButtonSlot(inventoryManager.getInventoryScreenButton(), actionButtonXOffset, ACTION_BUTTON_Y_OFFSET) {
			@Override
			public void onSlotClicked(ClickType clickType, int clickedButton, PlayerEntity player) {
				if(SlotManager.this.guiScreenHandler != null
						&& clickType == ClickType.PICKUP && player.inventory.getItemStack().isEmpty()) {
					SlotManager.this.guiScreenHandler.setGuiScreen(INVENTORY_SCREEN);
				}
			}
		};
		this.turtleContainerSC.addSlot(inventorySlot);

		actionButtonXOffset += ACTION_BUTTON_LENGTH;
		Slot terminalSlot = new ButtonSlot(inventoryManager.getTerminalButton(), actionButtonXOffset, ACTION_BUTTON_Y_OFFSET) {
			@Override
			public void onSlotClicked(ClickType clickType, int clickedButton, PlayerEntity player) {
				if(SlotManager.this.guiScreenHandler != null
						&& clickType == ClickType.PICKUP && player.inventory.getItemStack().isEmpty()) {
					SlotManager.this.guiScreenHandler.setGuiScreen(TERMINAL_SCREEN);
				}
			}
		};
		this.turtleContainerSC.addSlot(terminalSlot);
	}

	/**
	 * Updates the slot visibility based on the currently shown {@link TurtleInventoryScreen}.
	 * @param screen - The currently shown {@link TurtleInventoryScreen}.
	 */
	public void updateSlotVisibility(TurtleInventoryScreen screen) {
		
		// Hide all slots.
		for(List<InventorySlotManager> inventorySlotManagers : this.slotInventoryManagers.values()) {
			for(InventorySlotManager inventorySlotManager : inventorySlotManagers) {
				inventorySlotManager.setSlotsVisible(false);
			}
		}
		
		// Show the slots that should be shown on the screen.
		for(InventorySlotManager inventorySlotManager : this.slotInventoryManagers.get(screen)) {
			inventorySlotManager.setSlotsVisible(true);
		}
		
		// Show run button slot only in non-config mode.
		if(!this.tileTurtleSC.getInventoryManager().isConfigMode()) {
			this.runButtonSlot.xPos = RUN_BUTTON_X_OFFSET;
			this.runButtonSlot.yPos = RUN_BUTTON_Y_OFFSET;
		} else {
			this.runButtonSlot.xPos = INVISIBLE_LOCATION;
			this.runButtonSlot.yPos = INVISIBLE_LOCATION;
		}
		
		// Show garbage button only in the programming screen.
		if(screen == PROGRAMMING_SCREEN) {
			this.garbageButtonSlot.xPos = GARBAGE_BUTTON_X_OFFSET;
			this.garbageButtonSlot.yPos = GARBAGE_BUTTON_Y_OFFSET;
		} else {
			this.garbageButtonSlot.xPos = INVISIBLE_LOCATION;
			this.garbageButtonSlot.yPos = INVISIBLE_LOCATION;
		}
	}

	public void setGuiScreenHandler(GuiScreenHandler guiScreenHandler) {
		this.guiScreenHandler = guiScreenHandler;
	}

	public InventorySlotManager getProgrammingInventorySlotManager() {
		return this.programmingInventorySlotManager;
	}

	public InventorySlotManager getCodingInventorySlotManager() {
		return this.codingInventorySlotManager;
	}

	public Slot getRunButtonSlot() {
		return this.runButtonSlot;
	}
}
