package computercraftsc.client.gui.inventories;

import computercraftsc.client.gui.container.TurtleContainerSC;
import computercraftsc.client.gui.container.slot.CodingSlot;
import computercraftsc.client.gui.container.slotmanager.InventorySlotManager;
import computercraftsc.client.gui.inventories.config.CodeItemEntry;
import computercraftsc.shared.RegistrySC;
import computercraftsc.shared.items.ItemProgrammingIcon;
import computercraftsc.shared.turtle.core.code.compiler.CompileResult;
import computercraftsc.shared.turtle.core.code.compiler.TurtleLangCompiler;
import computercraftsc.shared.turtle.core.code.compiler.exception.SyntaxException;
import net.minecraft.client.Minecraft;
import net.minecraft.inventory.container.Container;
import net.minecraft.inventory.container.Slot;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;

import static computercraftsc.client.gui.GuiConstants.CODING_INVENTORY_COLUMNS;
import static computercraftsc.client.gui.GuiConstants.CODING_INVENTORY_VISIBLE_ROWS;
import static computercraftsc.client.gui.GuiConstants.CODING_INVENTORY_TOTAL_ROWS;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

public class CodingInventory extends Inventory {

	private final InventoryManager inventoryManager;
	private final ToggleButtonInventory playButtonInventory;
	private CompileResult compileResult = null;
	private Set<Integer> lockedInventorySlotIds = new HashSet<>();

	/**
	 * Dit is de Inventory die de Code Schrijft
	 * @param inventoryManager - The inventory manager containing this inventory.
	 * @param playButtonInventory - The play button inventory.
	 */
	public CodingInventory(InventoryManager inventoryManager, ToggleButtonInventory playButtonInventory) {
		super(CODING_INVENTORY_COLUMNS, CODING_INVENTORY_VISIBLE_ROWS, CODING_INVENTORY_COLUMNS * CODING_INVENTORY_TOTAL_ROWS);
		this.inventoryManager = inventoryManager;
		this.playButtonInventory = playButtonInventory;
	}

	/**
	 * Gets the set of locked inventory slot IDs. Locked slots cannot be changed by the player.
	 * Changes made to this set are applied to the data, but do keep in mind that this set has to be kept in sync
	 * between the client and server.
	 * @return The {@link Set} of locked inventory slot IDs.
	 */
	public Set<Integer> getLockedInventorySlotIds() {
		return this.lockedInventorySlotIds;
	}

	/**
	 * Sets the set of locked inventory slot IDs. Locked slots cannot be changed by the player.
	 * Keep in mind that this set has to be kept in sync between the client and server.
	 * @param lockedInventorySlotIds - The {@Link Set} of locked inventory slot IDs. Use an empty set to indicate that
	 * no inventory slots are locked.
	 */
	public void setLockedInventorySlotIds(Set<Integer> lockedInventorySlotIds) {
		this.lockedInventorySlotIds = lockedInventorySlotIds;
	}

	@Override
	public Slot createSlot(int index, int xPosition, int yPosition) {
		return new CodingSlot(this, index, xPosition, yPosition) {
			@Override
			public void onSlotChanged() {
				super.onSlotChanged();
				CodingInventory.this.compileCode();
			}
			@Override
			public void putStack(ItemStack stack) {
				
				// Prevent non-changing slot edits from calling onSlotChanged() and spamming code recompiles.
				ItemStack oldStack = this.inventory.getStackInSlot(index);
				if(!ItemStack.areItemStacksEqual(stack, oldStack)) {
					super.putStack(stack);
				}
			}
		};
	}

	/**
	 * Return all items from the coding inventory back to the programming inventory.
	 * @param clearLockedInvSlots - Whether the locked coding inventory slots should be cleared or not.
	 */
	public void clearFullProgram(boolean clearLockedInvSlots) {
		for(int i = 0; i < this.getSizeInventory(); i++) {
			if(clearLockedInvSlots || !this.lockedInventorySlotIds.contains(i)) {
				ItemStack itemStack = this.removeStackFromSlot(i);
				if(!itemStack.isEmpty()) {
					this.inventoryManager.getProgrammingInventory().returnItemStack(itemStack);
				}
			}
		}
		this.compileCode();
	}

	/**
	 * Gets the last code compile result.
	 * @return The {@link CompileResult}.
	 */
	public CompileResult getCompileResult() {
		return this.compileResult;
	}

	/**
	 * Clears the inventory, resets the play button state and clears the compile result.
	 */
	@Override
	public void clear() {
		super.clear();

		// Reset the play/error button state.
		this.playButtonInventory.showPrimaryImage(true);

		// Clear the compile result.
		this.compileResult = null;
	}

	@Override
	public ItemStack removeStackFromSlot(int i) {
		ItemStack stack = this.getStackInSlot(i);
		this.setInventorySlotContents(i, ItemStack.EMPTY);
		return stack;
	}

	public InventorySlotManager getCodingInventorySlotManager() {
		@SuppressWarnings("resource")
		Container container = Minecraft.getInstance().player.openContainer;
		if(container instanceof TurtleContainerSC) {
			return ((TurtleContainerSC) container).getSlotManager().getCodingInventorySlotManager();
		}
		return null;
	}

	public void scrollToRow(int topRow) {
		@SuppressWarnings("resource")
		Container container = Minecraft.getInstance().player.openContainer;
		if(container instanceof TurtleContainerSC) {
			((TurtleContainerSC) container).getSlotManager().getCodingInventorySlotManager().scrollToRow(topRow);
		}
	}

	/**
	 * Compiles the code in this inventory, caches the compile result and updates the play/error button state.
	 * This should be called when the inventory contents have changed.
	 * @see #getCompileResult()
	 */
	public void compileCode() {

		// Don't compile code in config mode.
		if (this.inventoryManager.isConfigMode()) {
			this.compileResult = null;
			return;
		}

		// Compile code.
		try {
			this.compileResult = TurtleLangCompiler.compile(this.inventory);
		} catch (SyntaxException e) {
			this.compileResult = new CompileResult(null, Arrays.asList(e));
		}

		// Set play/error button state.
		boolean hasCompileErrors = !this.compileResult.getCompileExceptions().isEmpty();
		this.playButtonInventory.showPrimaryImage(!hasCompileErrors);
	}

	/**
	 * Checks whether there are any compile errors for the code that's currently in the inventory.
	 * @return {@code true} if there are compile errors, {@code false} otherwise.
	 */
	public boolean codeHasCompileErrors() {
		return this.compileResult == null || !this.compileResult.getCompileExceptions().isEmpty();
	}

	public void initializeConfigItems(boolean inConfigMode) {
		if (!inConfigMode) {
			return;
		}

		this.clear();

		int index = 0;
		ProgrammingInventory programmingInventory = this.inventoryManager.getProgrammingInventory();
		for (CodeItemEntry codeItemEntry : programmingInventory.getCodeItemEntries()) {
			int amount = codeItemEntry.getAmount();
			while (amount > 0 && index < this.getSizeInventory()) {
				int stackAmount = (amount > this.getInventoryStackLimit() ? this.getInventoryStackLimit() : amount);
				ItemStack itemStack = programmingInventory.getProgrammingIconRetriever()
						.getItemStackFromIdAndAmount(codeItemEntry.getIconId(), stackAmount);
				this.setInventorySlotContents(index++, itemStack);
				amount -= stackAmount;
			}
		}
	}

	/**
	 * Appends the given {@link ItemStack} to this inventory with indentation where needed.
	 * @param itemStack - The stack to append.
	 * @return The index of the item in this inventory if the stack was successfully appended, -1 otherwise
	 * (e.g. if the stack does not contain an {@link ItemProgrammingIcon}, or if the item cannot be inserted after
	 * the last item in the inventory).
	 */
	public int appendStackWithSmartIndentation(ItemStack itemStack) {

		// Find the index of the last item in the inventory.
		int lastItemIndex = this.getSizeInventory() - 1;
		while (lastItemIndex >= 0 && this.inventory[lastItemIndex].isEmpty()) {
			lastItemIndex--;
		}

		// Return false if the item does not fit after the last item in the inventory, or if the item cannot be set.
		if (lastItemIndex >= this.getSizeInventory() - 1 || itemStack.isEmpty()
				|| !(itemStack.getItem() instanceof ItemProgrammingIcon)) {
			return -1;
		}

		// Place item in the first slot if the inventory is empty.
		if (lastItemIndex == -1) {
			this.setInventorySlotContents(0, itemStack);
			this.compileCode();
			return 0;
		}

		// Get the current indentation.
		int currentIndentation = 0;
		int currentRowStartIndex = (lastItemIndex / CODING_INVENTORY_COLUMNS) * CODING_INVENTORY_COLUMNS;
		for(int i = currentRowStartIndex; i < lastItemIndex && this.inventory[i].isEmpty(); i++) {
			currentIndentation++;
		}

		// Determine the index to put the item at.
		Item lastItem = this.inventory[lastItemIndex].getItem();
		int nextRowStartIndex = currentRowStartIndex + CODING_INVENTORY_COLUMNS;
		boolean shouldIncrementAfter = shouldIncrementIndentAfter((ItemProgrammingIcon) lastItem);
		boolean shouldDecrementBefore = shouldDecrementIndentBefore((ItemProgrammingIcon) itemStack.getItem());
		boolean shouldIncrementLine = shouldIncrementLineBefore((ItemProgrammingIcon) itemStack.getItem())
				|| shouldIncrementLineAfter((ItemProgrammingIcon) lastItem)
				|| (lastItemIndex + 1) % CODING_INVENTORY_COLUMNS == 0 || shouldIncrementAfter || shouldDecrementBefore;
		int index = (shouldIncrementLine ? nextRowStartIndex + currentIndentation : lastItemIndex + 1);
		if (shouldIncrementAfter != shouldDecrementBefore) {
			if (shouldIncrementAfter) {
				if ((index + 1) % CODING_INVENTORY_COLUMNS != 0) { // Only increment when not line-wrapping from the last index.
					index++;
				}
			} else { // If shouldDecrementBefore.
				if (currentIndentation > 0) { // Only decrement when there is indentation available.
					index--;
				}
			}
		}

		// Return false if the index is not within the inventory.
		if (index >= this.getSizeInventory()) {
			return -1;
		}

		// Set the item in the inventory and return success.
		this.setInventorySlotContents(index, itemStack);
		this.compileCode();
		return index;
	}

	private static boolean shouldDecrementIndentBefore(ItemProgrammingIcon item) {
		return item == RegistrySC.ModItems.END.get()
				|| item == RegistrySC.ModItems.ELSE_IF.get()
				|| item == RegistrySC.ModItems.ELSE.get();
	}

	private static boolean shouldIncrementIndentAfter(ItemProgrammingIcon item) {
		return item == RegistrySC.ModItems.DO.get()
				|| item == RegistrySC.ModItems.THEN.get()
				|| item == RegistrySC.ModItems.ELSE.get();
	}

	private static boolean shouldIncrementLineBefore(ItemProgrammingIcon item) {
		return item == RegistrySC.ModItems.REPEAT.get()
				|| item == RegistrySC.ModItems.WHILE.get()
				|| item == RegistrySC.ModItems.FOR.get()
				|| item == RegistrySC.ModItems.IF.get()
				|| shouldDecrementIndentBefore(item);
	}

	private static boolean shouldIncrementLineAfter(ItemProgrammingIcon item) {
		return item == RegistrySC.ModItems.END.get()
				|| shouldIncrementIndentAfter(item);
	}
}
