package computercraftsc.client.gui.widgets;

import static computercraftsc.client.gui.GuiConstants.SCROLL_BAR_WIDTH;

import com.mojang.blaze3d.matrix.MatrixStack;

import computercraftsc.client.gui.KeyCodes;

public class ScrollPaneWidget extends Widget implements TwoLayerStateDrawable {
	
	private final ScrollBarWidget scrollbar;
	private final Widget content;
	
	private int scrollTopOffset = 0;
	
	protected ScrollPaneWidget(int xOffset, int yOffset, int contentWidth, int contentHeight, Widget content) {
		super(xOffset, yOffset, contentWidth + SCROLL_BAR_WIDTH, contentHeight);
		this.scrollbar = new ScrollBarWidget(contentWidth, 0, SCROLL_BAR_WIDTH, contentHeight,
				(Float progress) -> this.setScrollProgress(progress),
				(WidgetButton btn) -> this.scrollUp(),
				(WidgetButton btn) -> this.scrollDown());
		this.scrollbar.setParent(this);
		if(!(content instanceof ScrollableContent)) {
			throw new IllegalArgumentException(
					"Content widget must be instanceof " + ScrollableContent.class.getSimpleName() + ".");
		}
		if(!(content instanceof TwoLayerStateDrawable)) {
			throw new IllegalArgumentException(
					"Content widget must be instanceof " + TwoLayerStateDrawable.class.getSimpleName() + ".");
		}
		this.content = content;
		content.setParent(this);
	}
	
	@Override
	public void setVisible(boolean visible) {
		super.setVisible(visible);
		this.scrollbar.setVisible(visible);
		this.content.setVisible(visible);
	}
	
	@Override
	public boolean mouseScrolled(double mouseX, double mouseY, double scrollDelta) {
		if(!this.isMouseOnWidget()) {
			return false;
		}
		if(scrollDelta > 0) {
			this.scrollUp();
		} else if(scrollDelta < 0) {
			this.scrollDown();
		}
		return true;
	}
	
	@Override
	public boolean mousePressed(int mouseX, int mouseY, int mouseButton) {
		return this.scrollbar.mousePressed(
				mouseX - this.scrollbar.getRelXPosition(), mouseY - this.scrollbar.getRelYPosition(), mouseButton);
	}
	
	@Override
	public boolean mouseReleased(int mouseX, int mouseY, int mouseButton) {
		return this.scrollbar.mouseReleased(
				mouseX - this.scrollbar.getRelXPosition(), mouseY - this.scrollbar.getRelYPosition(), mouseButton);
	}
	
	@Override
	public void mouseMoved(int mouseX, int mouseY) {
		this.scrollbar.mouseMoved(mouseX - this.scrollbar.getRelXPosition(), mouseY - this.scrollbar.getRelYPosition());
	}
	
	@Override
	public boolean keyPressed(int keyCode, int scanCode, int modifiers) {
		if(!this.isMouseOnWidget()) {
			return false;
		}
		if(keyCode == KeyCodes.ARROW_UP) {
			this.scrollUp();
			return true;
		} else if(keyCode == KeyCodes.ARROW_DOWN) {
			this.scrollDown();
			return true;
		}
		return false;
	}
	
	private void scrollUp() {
		this.scroll(-((ScrollableContent) this.content).getScrollHeightStep());
	}
	
	private void scrollDown() {
		this.scroll(((ScrollableContent) this.content).getScrollHeightStep());
	}
	
	private void scroll(int scrollHeightStep) {
		ScrollableContent scrollableContent = (ScrollableContent) this.content;
		
		// Add scroll height step.
		int oldScrollTopOffset = this.scrollTopOffset;
		this.scrollTopOffset += scrollHeightStep;
		
		// Limit scroll height step to content bounds.
		int maxScrollTopOffset = scrollableContent.getContentHeight() - scrollableContent.getVisibleContentHeight();
		if(this.scrollTopOffset > maxScrollTopOffset) {
			this.scrollTopOffset = maxScrollTopOffset;
		}
		if(this.scrollTopOffset < 0) {
			this.scrollTopOffset = 0;
		}
		
		// Notify content about the scroll change.
		if(this.scrollTopOffset != oldScrollTopOffset) {
			scrollableContent.onScrollTopOffsetChanged(this.scrollTopOffset);
		}
	}
	
	private void setScrollProgress(float progress) {
		int oldScrollTopOffset = this.scrollTopOffset;
		progress = (progress < 0 ? 0 : (progress > 1 ? 1 : progress));
		ScrollableContent scrollableContent = (ScrollableContent) this.content;
		int maxScrollTopOffset = scrollableContent.getContentHeight() - scrollableContent.getVisibleContentHeight();
		this.scrollTopOffset = (int) (progress * maxScrollTopOffset);
		
		// Notify content about the scroll change.
		if(this.scrollTopOffset != oldScrollTopOffset) {
			scrollableContent.onScrollTopOffsetChanged(this.scrollTopOffset);
		}
	}
	
	@Override
	public void drawGuiContainerBackgroundLayer(MatrixStack matrixStack, int mouseX, int mouseY) {
		this.scrollbar.drawGuiContainerBackgroundLayer(
				matrixStack, mouseX - this.scrollbar.getRelXPosition(), mouseY - this.scrollbar.getRelYPosition());
		((TwoLayerStateDrawable) this.content).drawGuiContainerBackgroundLayer(
				matrixStack, mouseX - this.content.getRelXPosition(), mouseY - this.content.getRelYPosition());
	}
	
	@Override
	public void drawGuiContainerForegroundLayer(MatrixStack matrixStack, int mouseX, int mouseY) {
		this.scrollbar.drawGuiContainerForegroundLayer(
				matrixStack, mouseX - this.scrollbar.getRelXPosition(), mouseY - this.scrollbar.getRelYPosition());
		((TwoLayerStateDrawable) this.content).drawGuiContainerForegroundLayer(
				matrixStack, mouseX - this.content.getRelXPosition(), mouseY - this.content.getRelYPosition());
		
		// Update scroll bar progress, limiting scroll height step to content bounds if necessary.
		ScrollableContent scrollableContent = (ScrollableContent) this.content;
		int contentScrollTopOffset = scrollableContent.getScrollTopOffset();
		if(contentScrollTopOffset >= 0) {
			this.scrollTopOffset = contentScrollTopOffset;
		}
		int maxScrollTopOffset = scrollableContent.getContentHeight() - scrollableContent.getVisibleContentHeight();
		if(this.scrollTopOffset > maxScrollTopOffset) {
			this.scrollTopOffset = maxScrollTopOffset;
		}
		if(this.scrollTopOffset < 0) {
			this.scrollTopOffset = 0;
		}
		this.scrollbar.updateBar(scrollableContent.getContentHeight(),
				scrollableContent.getVisibleContentHeight(), this.scrollTopOffset);
	}
	
	public Widget getContent() {
		return this.content;
	}
}
