package computercraftsc.client.gui.widgets;

import net.minecraft.client.Minecraft;
import net.minecraft.client.audio.SimpleSound;
import net.minecraft.client.gui.AbstractGui;
import net.minecraft.client.renderer.Tessellator;
import net.minecraft.client.renderer.vertex.DefaultVertexFormats;
import net.minecraft.util.SoundEvents;

import org.lwjgl.opengl.GL11;
import com.mojang.blaze3d.matrix.MatrixStack;
import com.mojang.blaze3d.systems.RenderSystem;

import computercraftsc.client.utils.MouseUtils;

public abstract class Widget extends AbstractGui {
	
	protected static final int LEFT_MOUSE_BTN = 0;
	protected static final int RIGHT_MOUSE_BTN = 1;
	protected static final int MIDDLE_MOUSE_BTN = 2;
	
	private Widget parent = null;
	private int inParentXPos;
	private int inParentYPos;
	private int width;
	private int height;
	private boolean isVisible = true;
	private int zLevel = 0;
	
	/**
	 * Creates a new {@link Widget}.
	 * Coordinates are the left top position relative to the parent left top position.
	 * @param x - The relative x coordinate.
	 * @param y - The relative y coordinate.
	 * @param width - The width.
	 * @param height - The height.
	 */
	public Widget(int x, int y, int width, int height) {
		this.inParentXPos = x;
		this.inParentYPos = y;
		this.width = width;
		this.height = height;
	}
	
	public void setParent(Widget parent) {
		this.parent = parent;
	}
	
	public boolean isVisible() {
		return this.isVisible && (this.parent == null || this.parent.isVisible());
	}
	
	public void setVisible(boolean visible) {
		this.isVisible = visible;
	}
	
	public int getRelXPosition() {
		return this.inParentXPos;
	}
	
	public int getRelYPosition() {
		return this.inParentYPos;
	}
	
	public int getAbsoluteXPosition() {
		return this.inParentXPos + (this.parent != null ? this.parent.getAbsoluteXPosition() : 0);
	}
	
	public int getAbsoluteYPosition() {
		return this.inParentYPos + (this.parent != null ? this.parent.getAbsoluteYPosition() : 0);
	}
	
	public int getWidth() {
		return this.width;
	}
	
	public int getHeight() {
		return this.height;
	}
	
	public void setPosition(int x, int y) {
		this.inParentXPos = x;
		this.inParentYPos = y;
	}
	
	public void resize(int width, int height) {
		this.width = width;
		this.height = height;
	}
	
	public void tick() {
	}
	
	/**
	 * Draws the {@link Widget} to the screen.
	 * The mouse coordinates are relative to the left top of this {@link Widget}.
	 * @param matrixStack
	 * @param mc
	 * @param xLeftTop - The absolute x coordinate of the left top.
	 * @param yLeftTop - The absolute y coordinate of the left top.
	 * @param mouseX - The mouse x position.
	 * @param mouseY - The mouse y position.
	 */
	public void draw(MatrixStack matrixStack, Minecraft mc, int xLeftTop, int yLeftTop, int mouseX, int mouseY) {
	}
	
	public void drawForeground(Minecraft mc, int xOrigin, int yOrigin, int mouseX, int mouseY) {
	}
	
	/**
	 * Called when the mouse is moved.
	 * The mouse coordinates are relative to the left top of this {@link Widget}.
	 * @param mouseX - The mouse x position.
	 * @param mouseY - The mouse y position.
	 */
	public void mouseMoved(int mouseX, int mouseY) {
	}
	
	/**
	 * Called when a mouse button is pressed.
	 * The mouse coordinates are relative to the left top of this {@link Widget}.
	 * @param mouseX - The mouse x position.
	 * @param mouseY - The mouse y position.
	 * @param mouseButton - The mouse button, being
	 * {@link #LEFT_MOUSE_BTN}, {@link #RIGHT_MOUSE_BTN} or {@link #MIDDLE_MOUSE_BTN}.
	 * @return {@code true} if the mouse press has been handled and should not be handled in other gui components,
	 * {@code false} otherwise.
	 */
	public boolean mousePressed(int mouseX, int mouseY, int mouseButton) {
		return false;
	}
	
	/**
	 * Called when a mouse button is released.
	 * The mouse coordinates are relative to the left top of this {@link Widget}.
	 * @param mouseX - The mouse x position.
	 * @param mouseY - The mouse y position.
	 * @param mouseButton - The mouse button, being
	 * {@link #LEFT_MOUSE_BTN}, {@link #RIGHT_MOUSE_BTN} or {@link #MIDDLE_MOUSE_BTN}.
	 * @return {@code true} if the mouse release has been handled and should not be handled in other gui components,
	 * {@code false} otherwise.
	 */
	public boolean mouseReleased(int mouseX, int mouseY, int mouseButton) {
		return false;
	}
	
	/**
	 * Called when the mouse is scrolled.
	 * The mouse coordinates are relative to the left top of this {@link Widget}.
	 * @param mouseX - The mouse x position.
	 * @param mouseY - The mouse y position.
	 * @param scrollDelta - The amount that was scrolled. Negative for down, positive for up.
	 * @return {@code true} if the mouse scroll has been handled and should not be handled in other gui components,
	 * {@code false} otherwise.
	 */
	public boolean mouseScrolled(double mouseX, double mouseY, double scrollDelta) {
		return false;
	}
	
	public boolean keyPressed(int keyCode, int scanCode, int modifiers) {
		return false;
	}
	
	public boolean charTyped(char ch, int modifiers) {
		return false;
	}
	
	protected void drawFullImage(int x, int y, int w, int h) {
		Tessellator tessellator = Tessellator.getInstance();
		tessellator.getBuffer().begin(GL11.GL_QUADS, DefaultVertexFormats.POSITION_TEX);
		tessellator.getBuffer().pos((double) (x + 0), (double) (y + h), (double) this.zLevel).tex(0.0f, 1.0f).endVertex();
		tessellator.getBuffer().pos((double) (x + w), (double) (y + h), (double) this.zLevel).tex(1.0f, 1.0f).endVertex();
		tessellator.getBuffer().pos((double) (x + w), (double) (y + 0), (double) this.zLevel).tex(1.0f, 0.0f).endVertex();
		tessellator.getBuffer().pos((double) (x + 0), (double) (y + 0), (double) this.zLevel).tex(0.0f, 0.0f).endVertex();
		tessellator.draw();
	}
	
	protected void drawT3(MatrixStack matrixStack, int x, int y, int w, int h, int u, int v, int uw, int vh) {
		int partW = uw / 3;
		
		// Draw first bit
		this.blit(matrixStack, x, y, u, v, partW, vh);
		
		// Draw middle bits
		int middleBits = Math.max((w - 2 * partW) / partW, 0);
		for(int j=0; j<middleBits; ++j) {
			this.blit(matrixStack, x + (j + 1) * partW, y, u + partW, v, partW, vh);
		}
		
		// Draw end bit
		int endW = w - (middleBits + 1) * partW;
		this.blit(matrixStack, x + w - endW, y, u + uw - endW, v, endW, vh);
	}
	
	protected void drawT9(MatrixStack matrixStack, int x, int y, int w, int h, int u, int v, int uw, int vh) {
		int partH = vh / 3;
		
		// Draw first row
		drawT3(matrixStack, x, y, w, partH, u, v, uw, partH);
		
		// Draw middle rows
		int middleBits = Math.max((h - 2 * partH) / partH, 0);
		for(int j=0; j<middleBits; ++j) {
			drawT3(matrixStack, x, y + (j + 1) * partH, w, partH, u, v + partH, uw, partH);
		}
		
		// Draw end row
		int endH = h - (middleBits + 1) * partH;
		this.drawT3(matrixStack, x, y + h - endH, w, endH, u, v + vh - endH, uw, endH);
	}
	
	protected void drawInsetBorder(MatrixStack matrixStack, int x, int y, int w, int h) {
		// Draw border
		try {
			this.vLine(matrixStack, x, y - 1, y + h - 1, 0xff363636);
			this.vLine(matrixStack, x + w - 1, y, y + h, 0xffffffff);
			this.hLine(matrixStack, x, x + w - 2, y, 0xff363636);
			this.hLine(matrixStack, x + 1, x + w - 1, y + h - 1, 0xffffffff);
			this.hLine(matrixStack, x, x, y + h - 1, 0xff8a8a8a);
			this.hLine(matrixStack, x + w - 1, x + w - 1, y, 0xff8a8a8a);
		} finally {
			RenderSystem.color4f(1.0f, 1.0f, 1.0f, 1.0f);
		}
	}
	
	protected void drawString(MatrixStack matrixStack, String s, int x, int y, int color) {
		Minecraft mc = Minecraft.getInstance();
		try {
			mc.fontRenderer.drawString(matrixStack, s, x, y, color);
		} finally {
			RenderSystem.color4f(1.0f, 1.0f, 1.0f, 1.0f);
		}
	}
	
	protected int getStringWidth(String s) {
		Minecraft mc = Minecraft.getInstance();
		return mc.fontRenderer.getStringWidth(s);
	}
	
	protected void playClickSound() {
		Minecraft mc = Minecraft.getInstance();
		mc.getSoundHandler().play(SimpleSound.master(SoundEvents.UI_BUTTON_CLICK, 1.0F));
	}
	
	/**
	 * Checks whether the mouse is on this {@link Widget} according to the absolute position, width and height.
	 * @return {@code true} if the mouse is within the bounds of this widget, {@code false} otherwise.
	 */
	protected boolean isMouseOnWidget() {
		MouseUtils.MousePos mousePos = MouseUtils.getMousePosition();
		double mouseX = mousePos.x;
		double mouseY = mousePos.y;
		
		return mouseX >= this.getAbsoluteXPosition()
				&& mouseX <= this.getAbsoluteXPosition() + this.getWidth()
				&& mouseY >= this.getAbsoluteYPosition()
				&& mouseY <= this.getAbsoluteYPosition() + this.getHeight();
	}
}
