package computercraftsc.client.gui.widgets;

import java.util.ArrayList;
import java.util.Iterator;

import computercraftsc.client.gui.KeyCodes;
import computercraftsc.shared.items.ItemProgrammingIcon;
import net.minecraft.item.Item;
import net.minecraft.util.registry.Registry;
import net.minecraft.util.text.TranslationTextComponent;

public class WidgetItemSelectionPopup extends WidgetPopup {

	public static final int MATCHING_ITEMS_LIMIT = 128;

	private IItemSelectionListener m_listener;
	private WidgetTextBox m_textBox;
	private WidgetListBox m_listBox;
	private ArrayList<Item> m_matchingItems;
	public WidgetItemSelectionPopup(int x, int y, Item defaultItem, IItemSelectionListener listener) {
		super(x, y, 180, 18);
		this.m_listener = listener;
		this.m_matchingItems = new ArrayList<Item>();
		WidgetContainer content = this.getContent();

		// Add "OK" button.
		final WidgetButton okButton = new WidgetTextButton("gui.computercraftsc:popup_ok",
				content.getWidth() - 24, 0, 24, (WidgetButton button) -> this.tryConfirm());
		content.addWidget(okButton);

		// Add item name input field.
		this.m_textBox = new WidgetTextBox(this.getItemName(defaultItem), 32, TextAlignment.LEFT,
				0, 0, content.getWidth() - 24 - 2, new WidgetTextBox.ITextBoxListener() {

					@Override
					public void onTextChanged(WidgetTextBox widget) {
						refresh(widget);
					}

					@Override
					public void onFocusChanged(WidgetTextBox widget) {
						refresh(widget);
					}

					@Override
					public boolean isTextValid(String text) {
						return true;
					}

					private void refresh(WidgetTextBox widget) {
						if(widget.getFocus()) {
							WidgetItemSelectionPopup.this.findMatchingItems(true);
							okButton.setLocked((WidgetItemSelectionPopup.this.m_listBox.getSelection() < 0));
						}
					}
				});
		content.addWidget(this.m_textBox);

		// Add search results list box.
		this.m_listBox = new WidgetListBox(0, this.m_textBox.getRelYPosition() + this.m_textBox.getHeight() + 2,
				content.getWidth(), 7, new String[0], new IListBoxListener() {

					@Override
					public void onSelectionChanged(WidgetListBox widget) {
						refresh(widget);
					}

					@Override
					public void onFocusChanged(WidgetListBox widget) {
						refresh(widget);
					}

					private void refresh(WidgetListBox widget) {
						okButton.setLocked((widget.getSelection() < 0));
						if(!WidgetItemSelectionPopup.this.m_textBox.getFocus()) {
							String selectedText = widget.getSelectedText();
							if(selectedText != null)
								WidgetItemSelectionPopup.this.m_textBox.setText(selectedText);
						}
					}
				}, true);
		content.addWidget(this.m_listBox);

		this.resizeContent(this.m_listBox.getWidth(), this.m_listBox.getRelYPosition() + this.m_listBox.getHeight());
		this.findMatchingItems(defaultItem != null);
		if(defaultItem != null) {
			for(int i = 0; i < this.m_matchingItems.size(); i++) {
				if(this.m_matchingItems.get(i) == defaultItem) {
					this.m_listBox.setSelection(i);
					break;
				}
			}
		}
		okButton.setLocked((this.m_listBox.getSelection() < 0));
		this.m_textBox.setFocus(true);
	}

	@Override
	public boolean keyPressed(int keyCode, int scanCode, int modifiers) {
		if(keyCode == KeyCodes.TAB) {
			if(this.m_textBox.getFocus()) {
				this.m_textBox.setFocus(false);
				this.m_listBox.setFocus(true);
				if(this.m_listBox.getSelection() < 0 && this.m_matchingItems.size() > 0) {
					this.m_listBox.setSelection(0);
					this.m_textBox.setText(this.m_listBox.getSelectedText());
				}
			} else if(this.m_listBox.getFocus()) {
				this.m_listBox.setFocus(false);
				this.m_textBox.setFocus(true);
			}
		}
		return super.keyPressed(keyCode, scanCode, modifiers);
	}

	@Override
	protected void onCancel() {
		this.m_listener.onCancel(this);
	}

	@Override
	protected void onConfirm() {
		if(tryConfirm())
			playClickSound();
	}

	private boolean tryConfirm() {
		Item selectedItem = getSelectedItem();
		if(selectedItem != null) {
			this.m_listener.onConfirmation(this, selectedItem);
			return true;
		}
		return false;
	}

	private Item getSelectedItem() {
		int selection = this.m_listBox.getSelection();
		if(selection >= 0)
			return this.m_matchingItems.get(selection);
		return null;
	}

	private String getItemName(Item item) {
		if(item != null) {
			return new TranslationTextComponent(item.getTranslationKey()).getString();
		}
		return "";
	}

	private void findMatchingItems(boolean selectFirstMatch) {
		this.m_matchingItems.clear();
		int match = -1;
		int i = 0;
		String filter = this.m_textBox.getText().toLowerCase();
		@SuppressWarnings("deprecation")
		Iterator<Item> iterator = Registry.ITEM.iterator();
		while (iterator.hasNext()) {
			Item item = iterator.next();
			if(item != null && !(item instanceof ItemProgrammingIcon)) {
				String name = getItemName(item).toLowerCase();
				if(name.contains(filter)) {
					this.m_matchingItems.add(item);
					if(name.equals(filter)) {
						match = i;
					}
					i++;
					if(this.m_matchingItems.size() >= MATCHING_ITEMS_LIMIT) {
						break;
					}
				}
			}
		}
		if(match < 0 && this.m_matchingItems.size() > 0) {
			match = 0;
		}
		String[] names = new String[this.m_matchingItems.size()];
		for(int j = 0; j < names.length; j++) {
			Item item = this.m_matchingItems.get(j);
			names[j] = getItemName(item);
		}
		this.m_listBox.setOptions(names);
		if(selectFirstMatch) {
			this.m_listBox.setSelection(match);
		}
	}
}
