package computercraftsc.client.gui.widgets;

import computercraftsc.SCGlobal;
import computercraftsc.client.gui.inventories.InventoryManager;
import computercraftsc.client.gui.TurtleInventoryScreen;
import computercraftsc.client.gui.container.slotmanager.SlotManager;
import net.minecraft.client.resources.I18n;
import net.minecraftforge.api.distmarker.Dist;
import net.minecraftforge.api.distmarker.OnlyIn;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.mojang.blaze3d.matrix.MatrixStack;

import static computercraftsc.client.gui.GuiConstants.BACKGROUND_IMAGE_HEIGHT;
import static computercraftsc.client.gui.GuiConstants.BACKGROUND_IMAGE_WIDTH;
import static computercraftsc.client.gui.GuiConstants.CODING_INVENTORY_HEIGHT;
import static computercraftsc.client.gui.GuiConstants.CODING_INVENTORY_WIDTH;
import static computercraftsc.client.gui.GuiConstants.SLOT_SIZE;
import static computercraftsc.client.gui.GuiConstants.GARBAGE_BUTTON_X_OFFSET;
import static computercraftsc.client.gui.GuiConstants.GARBAGE_BUTTON_Y_OFFSET;
import static computercraftsc.client.gui.GuiConstants.OFFSET_BETWEEN_INVENTORY_TABS;
import static computercraftsc.client.gui.GuiConstants.PLAYER_INVENTORY_HEIGHT;
import static computercraftsc.client.gui.GuiConstants.PLAYER_INVENTORY_WIDTH;
import static computercraftsc.client.gui.GuiConstants.PROGRAMMING_INVENTORY_HEIGHT;
import static computercraftsc.client.gui.GuiConstants.PROGRAMMING_INVENTORY_WIDTH;
import static computercraftsc.client.gui.GuiConstants.SPACE_BETWEEN_PLAYER_BAR_AND_INVENTORY;
import static computercraftsc.client.gui.GuiConstants.INVENTORY_TAB_HEIGHT;
import static computercraftsc.client.gui.GuiConstants.INVENTORY_TAB_WIDTH;
import static computercraftsc.client.gui.GuiConstants.INVENTORY_TAB_Y_OFFSET;
import static computercraftsc.client.gui.GuiConstants.TERMINAL_HEIGHT;
import static computercraftsc.client.gui.GuiConstants.TERMINAL_WIDTH;
import static computercraftsc.client.gui.GuiConstants.TURTLE_INVENTORY_HEIGHT;
import static computercraftsc.client.gui.GuiConstants.TURTLE_INVENTORY_WIDTH;
import static computercraftsc.client.gui.GuiConstants.TURTLE_INVENTORY_X_OFFSET;
import static computercraftsc.client.gui.GuiConstants.X_OFFSET_LEFT;
import static computercraftsc.client.gui.GuiConstants.Y_OFFSET_TOP;
import static computercraftsc.client.gui.GuiConstants.Y_OFFSET_TOP_INVENTORY_NAME;
import static computercraftsc.client.gui.GuiConstants.PROGRAMMING_INVENTORY_X_OFFSET;
import static computercraftsc.client.gui.GuiConstants.RUN_BUTTON_X_OFFSET;
import static computercraftsc.client.gui.GuiConstants.RUN_BUTTON_Y_OFFSET;
import static computercraftsc.client.gui.GuiConstants.SLOT_BORDER_WIDTH;
import static computercraftsc.client.gui.TurtleInventoryScreen.INVENTORY_SCREEN;
import static computercraftsc.client.gui.TurtleInventoryScreen.PROGRAMMING_SCREEN;
import static computercraftsc.client.gui.TurtleInventoryScreen.TERMINAL_SCREEN;

/**
 * This class created the widgets that are to be displayed on screen. It creates them in order, and adds them to the
 * Widget list. At every frame it check which Widgets are to be displayed, and what are to be hidden.
 * The ScrollBarWidget is created separately as it is to be passed to the TerminalCodeWriter so it can be updated.
 */
@OnlyIn(Dist.CLIENT)
public class WidgetManager implements TwoLayerStateDrawable {

	private final int guiLeft;
	private final int guiTop;
	private final Map<TurtleInventoryScreen, WidgetContainer> containers = new HashMap<>();
	private final InventoryManager inventoryManager;
	private final SlotManager slotManager;
	private TurtleInventoryScreen screen;

	/**
	 * The InventoryManager is passed to determine Heights and Width of the image, to be the same as the columns and rows the have.
	 * Keep in mind that this only stretches the image to the size, and does not in fact add extra slots to the image.
	 * @param inventoryManager InventoryManager to find the rows and columns
	 */
	public WidgetManager(int guiLeft, int guiTop, InventoryManager inventoryManager, SlotManager slotManager) {
		this.guiLeft = guiLeft;
		this.guiTop = guiTop;
		this.inventoryManager = inventoryManager;
		this.slotManager = slotManager;

		// Setup widget containers.
		this.setupProgrammingScreenWidgets();
		this.setupInventoryScreenWidgets();
		this.setupTerminalScreenWidgets();

		this.setGuiScreen(PROGRAMMING_SCREEN);
	}

	/**
	 * Setup programming screen widgets.
	 */
	private void setupProgrammingScreenWidgets() {

		// Setup coding widget. Its slots are set by SlotManager.
		CodingWidget codingWidget = new CodingWidget(0, 0,
				CODING_INVENTORY_WIDTH, CODING_INVENTORY_HEIGHT, this.inventoryManager.getCodingInventory());
		Widget codingContainerWidget = new ScrollPaneWidget(X_OFFSET_LEFT, Y_OFFSET_TOP,
				CODING_INVENTORY_WIDTH, CODING_INVENTORY_HEIGHT, codingWidget);

		// Setup programming widget. Its slots are set by SlotManager.
		ProgrammingWidget programmingWidget = new ProgrammingWidget(0, 0,
				PROGRAMMING_INVENTORY_WIDTH, PROGRAMMING_INVENTORY_HEIGHT, this.inventoryManager);
		Widget programmingContainerWidget = new ScrollPaneWidget(PROGRAMMING_INVENTORY_X_OFFSET, Y_OFFSET_TOP,
				PROGRAMMING_INVENTORY_WIDTH, PROGRAMMING_INVENTORY_HEIGHT, programmingWidget);

		// Setup tab indicator widget.
		Widget tabIndicatorWidget = new WidgetImage(SCGlobal.STATE_TAB_TEXTURE,
				0, INVENTORY_TAB_Y_OFFSET, INVENTORY_TAB_WIDTH, INVENTORY_TAB_HEIGHT);

		// Setup button background widgets.
		Widget garbageBtnBackgroundWidget = new WidgetImage(SCGlobal.EMPTY_SLOT_TEXTURE,
				GARBAGE_BUTTON_X_OFFSET - SLOT_BORDER_WIDTH, GARBAGE_BUTTON_Y_OFFSET - SLOT_BORDER_WIDTH,
				SLOT_SIZE, SLOT_SIZE);
		Widget runBtnBackgroundWidget = new WidgetImage(SCGlobal.EMPTY_SLOT_TEXTURE,
				RUN_BUTTON_X_OFFSET - SLOT_BORDER_WIDTH, RUN_BUTTON_Y_OFFSET - SLOT_BORDER_WIDTH,
				SLOT_SIZE, SLOT_SIZE);

		// Setup text widget.
		Widget codingTextWidget = new TextWidget(I18n.format("gui.computercraftsc:code"),
				X_OFFSET_LEFT, Y_OFFSET_TOP - Y_OFFSET_TOP_INVENTORY_NAME);
		Widget programmingTextWidget = new TextWidget(I18n.format("gui.computercraftsc:given_items"),
				PROGRAMMING_INVENTORY_X_OFFSET, Y_OFFSET_TOP - Y_OFFSET_TOP_INVENTORY_NAME);

		List<Widget> programmingWidgets = new ArrayList<>();
		programmingWidgets.add(codingContainerWidget);
		programmingWidgets.add(programmingContainerWidget);
		programmingWidgets.add(tabIndicatorWidget);
		programmingWidgets.add(garbageBtnBackgroundWidget);
		programmingWidgets.add(runBtnBackgroundWidget);
		programmingWidgets.add(codingTextWidget);
		programmingWidgets.add(programmingTextWidget);

		this.containers.put(PROGRAMMING_SCREEN, new WidgetContainer(
				this.guiLeft, this.guiTop, BACKGROUND_IMAGE_WIDTH, BACKGROUND_IMAGE_HEIGHT, programmingWidgets));
	}

	/**
	 * Setup inventory screen widgets.
	 */
	private void setupInventoryScreenWidgets() {
		List<Widget> inventoryWidgets = new ArrayList<>();

		// Setup player's inventory widget, it's slots are set by SlotManager.
		Widget playerInventoryWidget = new WidgetImage(SCGlobal.PLAYER_INVENTORY_GRID_TEXTURE,
				X_OFFSET_LEFT, Y_OFFSET_TOP,
				PLAYER_INVENTORY_WIDTH, PLAYER_INVENTORY_HEIGHT + SPACE_BETWEEN_PLAYER_BAR_AND_INVENTORY);
		inventoryWidgets.add(playerInventoryWidget);

		// Setup turtle's inventory widget, it's slots are set by SlotManager.
		Widget turtleInventoryWidget = new WidgetImage(SCGlobal.TURTLE_INVENTORY_GRID_TEXTURE,
				TURTLE_INVENTORY_X_OFFSET, Y_OFFSET_TOP,
				TURTLE_INVENTORY_WIDTH, TURTLE_INVENTORY_HEIGHT);
		inventoryWidgets.add(turtleInventoryWidget);

		// Setup tab indicator widget.
		Widget tabIndicatorWidget = new WidgetImage(SCGlobal.STATE_TAB_TEXTURE,
				OFFSET_BETWEEN_INVENTORY_TABS, INVENTORY_TAB_Y_OFFSET, INVENTORY_TAB_WIDTH, INVENTORY_TAB_HEIGHT);
		inventoryWidgets.add(tabIndicatorWidget);

		// Setup button background widgets.
		Widget runBtnBackgroundWidget = new WidgetImage(SCGlobal.EMPTY_SLOT_TEXTURE,
				RUN_BUTTON_X_OFFSET - SLOT_BORDER_WIDTH, RUN_BUTTON_Y_OFFSET - SLOT_BORDER_WIDTH,
				SLOT_SIZE, SLOT_SIZE);
		inventoryWidgets.add(runBtnBackgroundWidget);

		// Setup inventory label widgets.
		Widget playerInventoryTextWidget = new TextWidget(I18n.format("gui.computercraftsc:player_inventory"),
				X_OFFSET_LEFT, Y_OFFSET_TOP - Y_OFFSET_TOP_INVENTORY_NAME);
		inventoryWidgets.add(playerInventoryTextWidget);
		Widget turtleInventoryTextWidget = new TextWidget(I18n.format("gui.computercraftsc:robot_inventory"),
				TURTLE_INVENTORY_X_OFFSET, Y_OFFSET_TOP - Y_OFFSET_TOP_INVENTORY_NAME);
		inventoryWidgets.add(turtleInventoryTextWidget);

		// Setup inventory slot id labels (4x4 inventory grid having slot ids 0 to 15).
		final int textOffsetInSlot = 2;
		final int slotIdLabelColor = 0xfac71e;
		for(int y = 0; y < 4; y++) {
			int yText = Y_OFFSET_TOP + textOffsetInSlot + y * SLOT_SIZE;
			for(int x = 0; x < 4; x++) {
				int xText = TURTLE_INVENTORY_X_OFFSET + textOffsetInSlot + x * SLOT_SIZE;
				inventoryWidgets.add(new WidgetDepthWrapper(
						new TextWidget(Integer.toString(x + y * 4), xText, yText, slotIdLabelColor), 300));
			}
		}

		this.containers.put(INVENTORY_SCREEN, new WidgetContainer(
				this.guiLeft, this.guiTop, BACKGROUND_IMAGE_WIDTH, BACKGROUND_IMAGE_HEIGHT, inventoryWidgets));
	}

	/**
	 * Setup terminal screen widgets.
	 */
	private void setupTerminalScreenWidgets() {

		// Setup terminal widget, the displayed code is written by TerminalCodeWriter.
		TerminalWidget terminalWidget = new TerminalWidget(0, 0,
				TERMINAL_WIDTH, TERMINAL_HEIGHT, this.inventoryManager.getCodingInventory());
		Widget terminalContainerWidget = new ScrollPaneWidget(X_OFFSET_LEFT, Y_OFFSET_TOP,
				TERMINAL_WIDTH, TERMINAL_HEIGHT, terminalWidget);

		// Setup tab indicator widget.
		Widget tabIndicatorWidget = new WidgetImage(SCGlobal.STATE_TAB_TEXTURE,
				OFFSET_BETWEEN_INVENTORY_TABS * 2, INVENTORY_TAB_Y_OFFSET, INVENTORY_TAB_WIDTH, INVENTORY_TAB_HEIGHT);

		// Setup button background widgets.
		Widget runBtnBackgroundWidget = new WidgetImage(SCGlobal.EMPTY_SLOT_TEXTURE,
				RUN_BUTTON_X_OFFSET - SLOT_BORDER_WIDTH, RUN_BUTTON_Y_OFFSET - SLOT_BORDER_WIDTH,
				SLOT_SIZE, SLOT_SIZE);

		// Setup text widget.
		Widget terminalTextWidget = new TextWidget(I18n.format("gui.computercraftsc:code"),
				X_OFFSET_LEFT, Y_OFFSET_TOP - Y_OFFSET_TOP_INVENTORY_NAME);

		List<Widget> terminalWidgets = new ArrayList<>();
		terminalWidgets.add(terminalContainerWidget);
		terminalWidgets.add(tabIndicatorWidget);
		terminalWidgets.add(runBtnBackgroundWidget);
		terminalWidgets.add(terminalTextWidget);

		this.containers.put(TERMINAL_SCREEN, new WidgetContainer(
				this.guiLeft, this.guiTop, BACKGROUND_IMAGE_WIDTH, BACKGROUND_IMAGE_HEIGHT, terminalWidgets));
	}

	/**
	 * Sets the shown internal GUI screen within the GUI screen.
	 * @param newScreen - The {@link TurtleInventoryScreen} to switch to.
	 */
	public void setGuiScreen(TurtleInventoryScreen newScreen) {

		// Return if the screen did not change.
		if(this.screen == newScreen) {
			return;
		}

		this.screen = newScreen;
		this.slotManager.updateSlotVisibility(newScreen);

		for(WidgetContainer container : this.containers.values()) {
			container.setVisible(false);
		}
		this.containers.get(newScreen).setVisible(true);
	}

	public TurtleInventoryScreen getScreen() {
		return this.screen;
	}

	public WidgetContainer getActiveWidgetContainer() {
		return this.containers.get(this.screen);
	}

	@Override
	public void drawGuiContainerBackgroundLayer(MatrixStack matrixStack, int mouseX, int mouseY) {
		this.containers.get(this.screen).drawGuiContainerBackgroundLayer(matrixStack, mouseX, mouseY);
	}

	@Override
	public void drawGuiContainerForegroundLayer(MatrixStack matrixStack, int mouseX, int mouseY) {
		this.containers.get(this.screen).drawGuiContainerForegroundLayer(matrixStack, mouseX, mouseY);
	}
}
