package computercraftsc.client.gui.widgets;

import com.mojang.blaze3d.matrix.MatrixStack;
import com.mojang.blaze3d.systems.RenderSystem;

import computercraftsc.SCGlobal;
import computercraftsc.client.gui.KeyCodes;
import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.screen.Screen;
import net.minecraft.util.ResourceLocation;

public class WidgetTextBox extends Widget {
	
	public static final ResourceLocation WIDGETS_RESOURCE_LOCATION =
			new ResourceLocation(SCGlobal.MOD_ID.toLowerCase(), "textures/gui/widgets.png");
	
	private String text;
	private int maxTextLength;
	private TextAlignment textAlignment;
	private boolean isFocussed;
	private int cursorPos;
	private ITextBoxListener textBoxListener;
	
	private boolean cursorBlinkState = false;
	private int cursorBlinkTickCounter = 0;
	private static final int CURSOR_BLINK_DURATION_TICKS = 8;
	
	public WidgetTextBox(String text, int maximumTextLength,
			TextAlignment alignment, int x, int y, int width, ITextBoxListener listener) {
		super(x, y, width, 18);
		this.maxTextLength = maximumTextLength;
		this.text = this.clip(text);
		this.textAlignment = alignment;
		this.isFocussed = false;
		this.cursorPos = this.text.length();
		this.textBoxListener = listener;
	}
	
	public void setText(String text) {
		if(!text.equals(this.text)) {
			this.text = this.clip(text);
			this.cursorPos = Math.min(this.cursorPos, this.text.length());
			this.notifyTextChanged();
		}
	}
	
	public String getText() {
		return this.text;
	}
	
	public void setFocus(boolean focus) {
		if(this.isFocussed != focus) {
			this.isFocussed = focus;
			this.notifyFocusChanged();
		}
	}
	
	public boolean getFocus() {
		return this.isFocussed;
	}
	
	public boolean isTextValid() {
		return this.textBoxListener.isTextValid(this.text);
	}
	
	@Override
	public void tick() {
		
		// Update cursor blink state.
		if(this.cursorBlinkTickCounter++ >= CURSOR_BLINK_DURATION_TICKS) {
			this.cursorBlinkState = !this.cursorBlinkState;
			this.cursorBlinkTickCounter = 0;
		}
		
		super.tick();
	}
	
	@Override
	public boolean mousePressed(int mouseX, int mouseY, int mouseButton) {
		if(mouseX >= 0 && mouseX < this.getWidth() && mouseY >= 0 && mouseY < this.getHeight()) {
			this.setFocus(true);
			int textStart = (this.textAlignment == TextAlignment.LEFT) ? 3
					: ((getWidth() - this.getStringWidth(this.text)) / 2);
			int localX = mouseX - textStart;
			int cursor = 0;
			while(cursor < this.text.length()
					&& localX >= this.getStringWidth(this.text.substring(0, cursor + 1)) - 1) {
				cursor++;
			}
			this.cursorPos = cursor;
			return true;
		} else {
			this.setFocus(false);
			return false;
		}
	}
	
	private boolean tryAddText(String text, boolean force) {
		if(text.isEmpty()) {
			return true;
		}
		String newText = this.text.substring(0, this.cursorPos) + text + this.text.substring(this.cursorPos);
		if(force || (newText.length() < this.maxTextLength && this.getStringWidth(newText) <= this.getWidth() - 6)) {

			this.text = this.clip(newText);
			this.cursorPos = Math.min(this.cursorPos + text.length(), this.text.length());
			this.notifyTextChanged();
			return true;
		}
		return false;
	}
	
	@Override
	public boolean keyPressed(int keyCode, int scanCode, int modifiers) {
		if(this.isFocussed) {
			if(keyCode == KeyCodes.V && modifiers == KeyCodes.MODIFIER_CTRL_MASK) {
				@SuppressWarnings("resource")
				String clipboardText = Minecraft.getInstance().keyboardListener.getClipboardString();
				if(clipboardText != null) {
					
					// Cut text at \r or \n when present.
					for(int i = 0; i < clipboardText.length(); i++) {
						char ch = clipboardText.charAt(i);
						if(ch == '\r' || ch == '\n') {
							clipboardText = clipboardText.substring(0, i);
							break;
						}
					}
					
					// Remove invalid chars.
					for(int i = clipboardText.length() - 1; i >= 0; i--) {
						char ch = clipboardText.charAt(i);
						if(ch < 32 || ch > 126) {
							clipboardText = clipboardText.substring(0, i) + clipboardText.substring(i + 1);
						}
					}
					
					// Add text.
					if(this.text.length() + clipboardText.length() > this.maxTextLength) {
						clipboardText = clipboardText.substring(0, this.maxTextLength - this.text.length());
					}
					for(int i = clipboardText.length(); i > 0; i--) {
						if(this.tryAddText(clipboardText, false)) {
							break;
						}
						clipboardText = clipboardText.substring(0, i);
					}
				}
			} else {
				switch(keyCode) {
					case KeyCodes.BACKSPACE: {
						if(this.cursorPos > 0) {
							this.text = this.text.substring(0, this.cursorPos - 1)
									+ this.text.substring(this.cursorPos);
							this.cursorPos--;
							this.notifyTextChanged();
						}
						return true;
					}
					case KeyCodes.HOME: {
						this.cursorPos = 0;
						return true;
					}
					case KeyCodes.ARROW_LEFT: {
						if(Screen.hasShiftDown()) {
							this.cursorPos = 0;
						} else if(this.cursorPos > 0) {
							this.cursorPos--;
						}
						return true;
					}
					case KeyCodes.ARROW_RIGHT: {
						if(Screen.hasShiftDown()) {
							this.cursorPos = this.text.length();
						} else if(this.cursorPos < this.text.length()) {
							this.cursorPos++;
						}
						return true;
					}
					case KeyCodes.END: {
						this.cursorPos = this.text.length();
						return true;
					}
					case KeyCodes.NUM_DELETE:
					case KeyCodes.DELETE: {
						if(this.cursorPos < this.text.length()) {
							this.text = this.text.substring(0, this.cursorPos)
									+ this.text.substring(this.cursorPos + 1);
							this.notifyTextChanged();
						}
						return true;
					}
				}
			}
		}
		return true; // Absorb all key presses.
	}
	
	@Override
	public boolean charTyped(char ch, int modifiers) {
		if(this.isFocussed && ch >= 32 && ch <= 126) { // Chars 32-126 are all non-extended text chars.
			this.tryAddText(Character.toString(ch), false);
		}
		return true; // Absorb all key presses.
	}
	
	@Override
	public void draw(MatrixStack matrixStack, Minecraft mc, int xLeftTop, int yLeftTop, int mouseX, int mouseY) {
		
		mc.getTextureManager().bindTexture(WIDGETS_RESOURCE_LOCATION);
		this.drawT3(matrixStack, xLeftTop, yLeftTop, this.getWidth(), this.getHeight(), 94, 36, 36, 18);
		
		String text = this.getText();
		int textColour = this.isTextValid() ? 16777215 : 16733525;
		int textStartX = xLeftTop
				+ ((this.textAlignment == TextAlignment.LEFT) ? 3 : ((this.getWidth() - this.getStringWidth(text)) / 2));
		int textStartY = yLeftTop + 5;
		this.drawString(matrixStack, text, textStartX, textStartY, textColour);
		if(this.isFocussed) {
			int lengthBeforeCursor = this.getStringWidth(this.text.substring(0, this.cursorPos));
			if(this.cursorBlinkState) {
				if(this.cursorPos < text.length()) {
					try {
						this.vLine(matrixStack, textStartX + lengthBeforeCursor - 1, textStartY - 1, textStartY + 8 - 1,
								0xFF191919 | textColour);
					} finally {
						RenderSystem.color4f(1.0F, 1.0F, 1.0F, 1.0F);
					}
				} else if(this.cursorPos == text.length()) {
					this.drawString(matrixStack, "_", textStartX + lengthBeforeCursor, textStartY, textColour);
				}
			}
		}
	}
	
	private void notifyTextChanged() {
		if(this.textBoxListener != null) {
			this.textBoxListener.onTextChanged(this);
		}
	}
	
	private void notifyFocusChanged() {
		if(this.textBoxListener != null) {
			this.textBoxListener.onFocusChanged(this);
		}
	}
	
	private String clip(String text) {
		if(text.length() > this.maxTextLength) {
			return text.substring(0, this.maxTextLength);
		}
		return text;
	}
	
	public static interface ITextBoxListener {
		void onTextChanged(WidgetTextBox widgetTextBox);
		void onFocusChanged(WidgetTextBox widgetTextBox);
		boolean isTextValid(String text);
	}
}
