package computercraftsc.shared.turtle.core.code.ast;

import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

import computercraftsc.shared.turtle.core.code.compiler.exception.CompileException;
import computercraftsc.shared.turtle.core.code.compiler.exception.TypeException;
import computercraftsc.shared.turtle.core.code.type.ClassType;
import net.minecraft.item.ItemStack;

public abstract class ASTTerm {
	
	private final int codeStartIndex;
	private final int codeEndIndex;
	
	public ASTTerm(int codeStartIndex, int codeEndIndex) {
		this.codeStartIndex = codeStartIndex;
		this.codeEndIndex = codeEndIndex;
	}
	
	public int getCodeStartIndex() {
		return this.codeStartIndex;
	}
	
	public int getCodeEndIndex() {
		return this.codeEndIndex;
	}
	
	/**
	 * Converts this AST term to its code representation.
	 * @return The code representation.
	 */
	public abstract String toCodeString();
	
	public abstract ClassType typecheck(List<CompileException> compileExceptions);
	
	protected void requireType(ASTTerm term, ClassType expectedType, List<CompileException> compileExceptions) {
		ClassType type = term.typecheck(compileExceptions);
		if(!type.isInstanceOf(expectedType) && type != ClassType.DYN && expectedType != ClassType.ANY) {
			compileExceptions.add(new TypeException(expectedType, type,
					term.getCodeStartIndex(), term.getCodeEndIndex()));
		}
	}
	
	/**
	 * Gets the next non-empty {@link ItemStack},
	 * setting the passed codeIndex to point at the stack if such a stack exists.
	 * The item at the passed index is not considered, so this method can be called multiple times to get multiple
	 * consecutive non-null items.
	 * @param code - The code {@link List}.
	 * @param codeIndex - The code index.
	 * @return The next non-empty {@link ItemStack}, or {@code null} when no such stack exists.
	 */
	protected static ItemStack getNextNonNullItem(List<ItemStack> code, AtomicInteger codeIndex) {
		for(int index = codeIndex.get() + 1; index < code.size(); index++) {
			ItemStack stack = code.get(index);
			if(!stack.isEmpty()) {
				codeIndex.set(index);
				return stack;
			}
		}
		return null;
	}
	
	/**
	 * Gets the first non-empty {@link ItemStack} after the given codeIndex (inclusive),
	 * setting the given codeIndex to point at the stack if such a stack exists.
	 * @param code - The code {@link List}.
	 * @param codeIndex - The code index.
	 * @return The next non-empty {@link ItemStack}, or {@code null} when no such stack exists.
	 */
	protected static ItemStack getFirstNonNullItem(List<ItemStack> code, AtomicInteger codeIndex) {
		for(int index = codeIndex.get(); index < code.size(); index++) {
			ItemStack stack = code.get(index);
			if(!stack.isEmpty()) {
				codeIndex.set(index);
				return stack;
			}
		}
		return null;
	}
	
	/**
	 * Skips over empty items. Starts checking at the given codeIndex.
	 * If no such item exists, the code index will be set to {@code code.size()}.
	 * @param code - The code {@link List}.
	 * @param codeIndex - The code index.
	 */
	protected static void skipEmptyItems(List<ItemStack> code, AtomicInteger codeIndex) {
		for(int index = codeIndex.get(); index < code.size(); index++) {
			ItemStack stack = code.get(index);
			if(!stack.isEmpty()) {
				codeIndex.set(index);
				return;
			}
		}
		codeIndex.set(code.size());
		return;
	}
}
