package computercraftsc.client.gui.inventories;

import net.minecraft.entity.player.PlayerEntity;
import net.minecraft.inventory.IInventory;
import net.minecraft.inventory.container.Slot;
import net.minecraft.item.ItemStack;

/**
 * The CustomAbstractInventory is used to create all the different inventories used in the Inventory.
 * Actions and parameters that are used in every inventory are defined here
 */
public abstract class Inventory implements IInventory {

	protected int columns;
	protected int visibleRows;
	protected int totalSlots;
	protected int maxSlotsShown;
	protected ItemStack[] inventory;

	public Inventory(int columns, int visibleRows, int totalSlots) {
		this.columns = columns;
		this.visibleRows = visibleRows;
		this.totalSlots = totalSlots;
		this.maxSlotsShown = columns * visibleRows;
		this.inventory = new ItemStack[totalSlots];
		for(int i = 0; i < totalSlots; i++) {
			this.inventory[i] = ItemStack.EMPTY;
		}
	}

	/**
	 * If Inventory isn't scrollable, it has a totalSlots of columns * rows
	 * @param columns columns
	 * @param rows rows
	 */
	public Inventory(int columns, int rows) {
		this(columns, rows, columns * rows);
	}

	public int getColumns() {
		return this.columns;
	}

	public int getVisibleRows() {
		return this.visibleRows;
	}

	/**
	 * Creates a new {@link Slot} for this inventory.
	 * @param index - The index of the slot in this inventory.
	 * @param xPosition - The screen x coordinate.
	 * @param yPosition - The screen y coordinate.
	 * @return The new {@link Slot}.
	 */
	public Slot createSlot(int index, int xPosition, int yPosition) {
		return new Slot(this, index, xPosition, yPosition);
	}

	@Override
	public int getSizeInventory() {
		return this.totalSlots;
	}

	public int getMaxSlotsShown() {
		return this.maxSlotsShown;
	}

	@Override
	public ItemStack getStackInSlot(int index) {
		if (index < 0 || index >= this.getSizeInventory()) {
			return ItemStack.EMPTY;
		}
		return this.inventory[index];
	}

	/**
	 * {@inheritDoc}
	 * This implementation ignores invalid indices and limits the given {@link ItemStack} according to the return value
	 * of {@link #getInventoryStackLimit()}.
	 */
	@Override
	public void setInventorySlotContents(int index, ItemStack stack) {

		// Skip out of bounds indices.
		if (index < 0 || index >= this.getSizeInventory()) {
			return;
		}

		// Respect inventory max stack size.
		if (stack != null && stack.getCount() > this.getInventoryStackLimit()) {
			stack.setCount(this.getInventoryStackLimit());
		}

		// Set item in inventory.
		this.inventory[index] = stack;
	}

	@Override
	public ItemStack decrStackSize(int i, int j) {
		ItemStack existing = this.getStackInSlot(i);
		if (!existing.isEmpty()) {
			if (existing.getCount() <= j) {
				this.setInventorySlotContents(i, ItemStack.EMPTY);
				return existing;
			} else {
				ItemStack part = existing.split(j);
				this.setInventorySlotContents(i, (existing.isEmpty() ? ItemStack.EMPTY : existing));
				return part;
			}
		} else {
			return ItemStack.EMPTY;
		}
	}

	@Override
	public ItemStack removeStackFromSlot(int i) {
		ItemStack stack = this.getStackInSlot(i);
		this.setInventorySlotContents(i, ItemStack.EMPTY);
		return stack;
	}

	public ItemStack[] getInventory() {
		return this.inventory;
	}

	@Override
	public int getInventoryStackLimit() {
		return 64;
	}

	@Override
	public void markDirty() {
	}

	@Override
	public boolean isUsableByPlayer(PlayerEntity player) {
		return true;
	}

	@Override
	public void openInventory(PlayerEntity player) {
	}

	@Override
	public void closeInventory(PlayerEntity player) {
	}

	@Override
	public boolean isItemValidForSlot(int i, ItemStack stack) {
		return true;
	}

	@Override
	public void clear() {
		int size = this.getSizeInventory();
		for (int i = 0; i < size; ++i) {
			this.inventory[i] = ItemStack.EMPTY;
		}
	}

	@Override
	public boolean isEmpty() {
		for(ItemStack stack : this.inventory) {
			if(!stack.isEmpty()) {
				return false;
			}
		}
		return true;
	}
}
