package computercraftsc.client.gui.inventories;

import computercraftsc.client.gui.ProgrammingIconRetriever;
import computercraftsc.client.gui.inventories.config.CodeItemEntry;
import computercraftsc.shared.RegistrySC;
import computercraftsc.shared.items.ItemProgrammingIcon;
import net.minecraft.entity.player.PlayerEntity;
import net.minecraft.entity.player.PlayerInventory;
import net.minecraft.item.ItemStack;
import org.apache.commons.lang3.ArrayUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

/**
 * @author Michiel
 * Deze Class verzamelt de verschillende inventorys die deel uitmaken van de het Turtle Systeem en handelt ze van een hoger niveau.
 * Op deze manier kan de logica worden gedelgeert naar de goede inventory en de passen acties worden uitgevoerd.
 */
public class InventoryManager {
	private final CodingInventory codingInventory;
	private final ProgrammingInventory programmingInventory;
	private final ToggleButtonInventory playButton;
	private final ActionButtonInventory closeButton;
	private final ActionButtonInventory garbageButton;
	private final ProgrammingIconRetriever programmingIconRetriever;
	private final ActionButtonInventory inventoryScreenButton;
	private final TurtleInventory turtleInventory;
	private final CCSCPlayerInventory playerInventory;
	private final ActionButtonInventory terminalScreenButton;
	private final ActionButtonInventory programmingScreenButton;
	private final List<Inventory> inventories;
	private boolean configMode = false;

	public CodingInventory getCodingInventory() {
		return this.codingInventory;
	}

	public ProgrammingInventory getProgrammingInventory() {
		return this.programmingInventory;
	}

	public ToggleButtonInventory getPlayButton() {
		return this.playButton;
	}

	public ActionButtonInventory getGarbageButton() {
		return this.garbageButton;
	}

	public ActionButtonInventory getInventoryScreenButton() {
		return this.inventoryScreenButton;
	}

	public TurtleInventory getTurtleInventory() {
		return this.turtleInventory;
	}

	public CCSCPlayerInventory getPlayerInventory() {
		return this.playerInventory;
	}

	public ActionButtonInventory getTerminalButton() {
		return this.terminalScreenButton;
	}

	public ActionButtonInventory getProgrammingScreenButton() {
		return this.programmingScreenButton;
	}

	public InventoryManager(ProgrammingIconRetriever programmingIconRetriever) {
		this.programmingIconRetriever = programmingIconRetriever;

		this.playButton = new ToggleButtonInventory(
				new ItemStack(RegistrySC.ModItems.BUTTON_RUN.get()),
				new ItemStack(RegistrySC.ModItems.BUTTON_STOP.get()));
		this.closeButton = new ActionButtonInventory(new ItemStack(RegistrySC.ModItems.BUTTON_CLOSE.get()));
		this.garbageButton = new ActionButtonInventory(new ItemStack(RegistrySC.ModItems.BUTTON_GARBAGE.get()));
		this.programmingScreenButton = new ActionButtonInventory(
				new ItemStack(RegistrySC.ModItems.BUTTON_PROGRAMMING_SCREEN.get()));
		this.inventoryScreenButton = new ActionButtonInventory(
				new ItemStack(RegistrySC.ModItems.BUTTON_TURTLE_INVENTORY_SCREEN.get()));
		this.terminalScreenButton = new ActionButtonInventory(
				new ItemStack(RegistrySC.ModItems.BUTTON_TERMINAL_SCREEN.get()));

		this.codingInventory = new CodingInventory(this, this.playButton);
		this.programmingInventory = new ProgrammingInventory(this, this.programmingIconRetriever);
		this.playerInventory = new CCSCPlayerInventory();
		this.turtleInventory = new TurtleInventory();

		this.inventories = new ArrayList<>();
		this.inventories.add(this.turtleInventory);
		this.inventories.add(this.playerInventory);
		this.inventories.add(this.codingInventory);
		this.inventories.add(this.programmingInventory);
		this.inventories.add(this.closeButton);
		this.inventories.add(this.playButton);
		this.inventories.add(this.programmingScreenButton);
		this.inventories.add(this.inventoryScreenButton);
		this.inventories.add(this.terminalScreenButton);
	}

	public int getSizeInventory() {
		int totalSlots = 0;
		for (Inventory inventory : this.inventories) {
			totalSlots += inventory.getSizeInventory();
		}
		return totalSlots;
	}

	public boolean isEmpty() {
		for(Inventory inventory : this.inventories) {
			if(!inventory.isEmpty()) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Loop door de Inventories en kies de inventory die bij de ID hoort geen geeft an de stack van de Slot terug
	 *
	 * @param index Index van de slot waar je op klikt
	 * @return ItemStack van de inventory
	 */
	public ItemStack getStackInSlot(int index) {
		for (Inventory inventory : this.inventories) {
			int size = inventory.getSizeInventory();
			if (index < size) {
				return inventory.getStackInSlot(index);
			}
			index -= size;
		}
		return ItemStack.EMPTY;
	}

	public ItemStack removeStackFromSlot(int index) {
		ItemStack stack = this.getStackInSlot(index);
		this.setInventorySlotContents(index, ItemStack.EMPTY);
		return stack;
	}

	/**
	 * Loopt door de Inventories en kies de inventory die bij de ID hoort en haalt het aantal items van de die plaats af
	 *
	 * @param index Index van de slot waar je op klikt
	 * @param count Aantal Items die ze van de Stack afhalen
	 * @return ItemStack van de inventory
	 */
	public ItemStack decrStackSize(int index, int count) {
		for (Inventory inventory : this.inventories) {
			int size = inventory.getSizeInventory();
			if (index < size) {
				return inventory.decrStackSize(index, count);
			}
			index -= size;
		}
		return ItemStack.EMPTY;
	}

	/**
	 * Loop door de Inventories en kies de inventory die bij de ID hoort en plaats de ItemStack op de goede plek
	 *
	 * @param index Index van de slot waar je op klikt
	 * @param stack ItemStakc die je erin plaats
	 */
	public void setInventorySlotContents(int index, ItemStack stack) {
		for (Inventory inventory : this.inventories) {
			int size = inventory.getSizeInventory();
			if (index < size) {
				inventory.setInventorySlotContents(index, stack);
				return;
			}
			index -= size;
		}
	}

	/**
	 * Gets the amount of each code item in the given inventory.
	 * @param inventory         array van item stack
	 * @return HashMap<index_of_icon, amount>
	 */
	private Map<Integer, Integer> getInventoryCodeItemCount(ItemStack[] inventory) {
		Map<Integer, Integer> iconsInInventory = new HashMap<>();
		for(ItemStack stack : inventory) {
			if(!stack.isEmpty() && stack.getItem() instanceof ItemProgrammingIcon) {
				ItemProgrammingIcon item = (ItemProgrammingIcon) stack.getItem();
				int index = this.programmingIconRetriever.getProgrammingIconId(item);
				int amount = stack.getCount();
				int storedAmount = iconsInInventory.getOrDefault(index, 0);
				iconsInInventory.put(index, storedAmount + amount);
			}
		}
		return iconsInInventory;
	}

	/**
	 * Corrects the programming inventory contents based on the code item entries
	 * ({@link ProgrammingInventory#getCodeItemEntries()}),
	 * the item stack held on the mouse, and the items in the coding inventory.
	 * @param heldItem - The item stack currently held by the mouse, or {@link ItemStack#EMPTY} if none.
	 */
	private void correctProgrammingInventoryContents(ItemStack heldItem) {

		// Do not update the programming inventory in config mode.
		if(this.configMode) {
			return;
		}

		// Get amount of code items per code item id in the coding inventory and on the mouse.
		ItemStack[] usedCodeItems = ArrayUtils.addAll(this.codingInventory.getInventory(), heldItem);
		Map<Integer, Integer> usedCodeItemIdCountMap = this.getInventoryCodeItemCount(usedCodeItems);

		// Insert zero-count placeholder entries for code items in the programming inventory.
		List<CodeItemEntry> codeItemEntries = this.programmingInventory.getCodeItemEntries();
		for(CodeItemEntry codeItemEntry : codeItemEntries) {
			if(!usedCodeItemIdCountMap.containsKey(codeItemEntry.getIconId())) {
				usedCodeItemIdCountMap.put(codeItemEntry.getIconId(), 0);
			}
		}

		// For all programming inventory code entries, insert the items that are not in use elsewhere.
		for(int invIndex = 0; invIndex < codeItemEntries.size(); invIndex++) {
			CodeItemEntry codeItemEntry = codeItemEntries.get(invIndex);
			int iconId = codeItemEntry.getIconId();
			int usedCodeItemCount = usedCodeItemIdCountMap.get(iconId);
			int numUnusedItems = codeItemEntry.getAmount() - usedCodeItemCount;

			// Set the item stack in the programming inventory.
			this.programmingInventory.setInventorySlotContents(invIndex,
					this.programmingIconRetriever.getItemStackFromIdAndAmount(iconId,
							(this.getProgrammingInventory().getHasInfiniteCodeItems() ? 1 : numUnusedItems)));
		}
	}

	public void setConfigMode(boolean configMode) {
		this.configMode = configMode;
	}

	public boolean isConfigMode() {
		return this.configMode;
	}

	public void setCodeItemEntries(List<CodeItemEntry> codeItemEntries) {
		this.programmingInventory.setCodeItemEntries(codeItemEntries);
	}


	/**
	 * Merges the code item entries in the programming inventory and the current code items in the coding inventory
	 * into a single list of code item entries. This is useful when using the coding inventory as a place to setup the
	 * code item entries.
	 * @return The list of code item entries sorted on id.
	 */
	public List<CodeItemEntry> getNewConfigItems() {
		List<CodeItemEntry> newCodeItemEntries = new ArrayList<>();
		ItemStack[] codingInventory = this.codingInventory.getInventory();
		Map<Integer, Integer> inventorySet = this.getInventoryCodeItemCount(codingInventory);
		for(Entry<Integer, Integer> entry : inventorySet.entrySet()) {
			newCodeItemEntries.add(new CodeItemEntry(entry.getKey(), entry.getValue()));
		}
		Collections.sort(newCodeItemEntries);
		return newCodeItemEntries;
	}

	public ActionButtonInventory getCloseButton() {
		return this.closeButton;
	}

	/**
	 * Initializes this manager and its inventories to be ready for usage.
	 * This method should be called when the GUI containing this inventory is being opened server-sided,
	 * before the GUI open request is sent to the player.
	 * @param player - The player opening the inventory.
	 * @param configMode - If {@code true}, the inventory will be opened in config mode.
	 */
	public void onGUIOpen(PlayerEntity player, boolean configMode) {
		this.configMode = configMode;
		this.codingInventory.initializeConfigItems(this.configMode);
		this.programmingInventory.initializeItems(this.configMode);
		this.codingInventory.compileCode();
		this.correctProgrammingInventoryContents(ItemStack.EMPTY);
	}

	public void setInventoryPlayer(PlayerInventory inventory) {
		this.playerInventory.setInventoryPlayer(inventory);
	}
}
