package computercraftsc.client.gui.inventories;

import computercraftsc.client.gui.ProgrammingIconRetriever;
import computercraftsc.client.gui.container.GuiScreenHandler;
import computercraftsc.client.gui.container.TurtleContainerSC;
import computercraftsc.client.gui.container.slot.ProgrammingSlot;
import computercraftsc.client.gui.container.slotmanager.InventorySlotManager;
import computercraftsc.client.gui.inventories.config.CodeItemEntry;
import computercraftsc.shared.items.ItemProgrammingIcon;
import net.minecraft.inventory.container.Container;
import net.minecraft.inventory.container.Slot;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;

import java.util.ArrayList;
import java.util.List;

import static computercraftsc.client.gui.GuiConstants.PROGRAMMING_INVENTORY_COLUMNS;
import static computercraftsc.client.gui.GuiConstants.PROGRAMMING_INVENTORY_ROWS;

public class ProgrammingInventory extends Inventory {

	private final InventoryManager inventoryManager;
	private final ProgrammingIconRetriever iconRetriever;
	private List<CodeItemEntry> codeItemEntries = new ArrayList<>();
	private boolean hasInfiniteCodeItems = false;
	private GuiScreenHandler containerHandler = null;

	public ProgrammingInventory(InventoryManager inventoryManager, ProgrammingIconRetriever iconRetriever) {
		super(PROGRAMMING_INVENTORY_COLUMNS, PROGRAMMING_INVENTORY_ROWS, (iconRetriever.getOrderedProgrammingIcons().size()
				+ PROGRAMMING_INVENTORY_COLUMNS - 1) / PROGRAMMING_INVENTORY_COLUMNS * PROGRAMMING_INVENTORY_COLUMNS);
		this.inventoryManager = inventoryManager;
		this.iconRetriever = iconRetriever;
	}

	@Override
	public Slot createSlot(int index, int xPosition, int yPosition) {
		return new ProgrammingSlot(this, index, xPosition, yPosition) {
			@Override
			public void onSlotChanged() {

				// Restore stack size to one in config or infinite code items mode.
				// Note that getting the item on a zero-count item stack will be air, so that should not be checked.
				if (ProgrammingInventory.this.inventoryManager.isConfigMode()
						|| ProgrammingInventory.this.hasInfiniteCodeItems) {
					this.getStack().setCount(1);
				}

				super.onSlotChanged();
			}
		};
	}

	public void setCodeItemEntries(List<CodeItemEntry> codeItemEntries) {
		this.codeItemEntries = codeItemEntries;
	}

	public List<CodeItemEntry> getCodeItemEntries() {
		return this.codeItemEntries;
	}

	public void setHasInfiniteCodeItems(boolean hasInfiniteCodeItems) {
		this.hasInfiniteCodeItems = hasInfiniteCodeItems;
	}

	public boolean getHasInfiniteCodeItems() {
		return this.hasInfiniteCodeItems;
	}

	@Override
	public ItemStack decrStackSize(int index, int amount) {

		// Return null if the inventory slot is empty.
		ItemStack existing = this.getStackInSlot(index);
		if (existing == null) {
			return null;
		}

		// Duplicate and return the requested amount of the clicked item in config mode.
		if (this.inventoryManager.isConfigMode() || this.hasInfiniteCodeItems) {
			return new ItemStack(existing.getItem(), amount);
		}

		// Decrease the stack size as usual, but leave a zero-size item stack if the slot has been emptied.
		ItemStack ret = super.decrStackSize(index, amount);
		ItemStack newExisting = this.getStackInSlot(index);
		if (newExisting == null) {
			this.setInventorySlotContents(index, new ItemStack(existing.getItem(), 0));
		}
		return ret;
	}

	@Override
	public ItemStack removeStackFromSlot(int index) {
		if (this.inventoryManager.isConfigMode() || this.hasInfiniteCodeItems) {
			ItemStack existing = this.getStackInSlot(index);
			if (existing != null) {
				return new ItemStack(existing.getItem(), 1);
			} else {
				return null;
			}
		} else {
			return super.removeStackFromSlot(index);
		}
	}

	@Override
	public int getInventoryStackLimit() {
		if (this.inventoryManager.isConfigMode() || this.hasInfiniteCodeItems) {
			return 1;
		} else {
			return 64;
		}
	}

	/**
	 * Initializes this programming inventory with either all items in config mode, or with only the configured items
	 * with the configured amounts. If this programming inventory is set to have infinite items, then one of each
	 * configured item is placed.
	 * @param inConfigMode - {@code true} if this inventory is in config mode, {@code false} otherwise.
	 */
	public void initializeItems(boolean inConfigMode) {
		this.clear();
		if(inConfigMode) {
			List<ItemProgrammingIcon> orderedProgrammingItems =
					this.getProgrammingIconRetriever().getOrderedProgrammingIcons();
			for(int i = 0; i < orderedProgrammingItems.size(); i++) {
				ItemProgrammingIcon item = orderedProgrammingItems.get(i);
				this.setInventorySlotContents(i, new ItemStack(item, 1));
			}
		} else {
			for(int i = 0; i < this.codeItemEntries.size(); i++) {
				CodeItemEntry codeItemEntry = this.codeItemEntries.get(i);
				ItemStack stack = this.getProgrammingIconRetriever().getItemStackFromIdAndAmount(
						codeItemEntry.getIconId(), (this.hasInfiniteCodeItems ? 1 : codeItemEntry.getAmount()));
				this.setInventorySlotContents(i, stack);
			}
		}
		this.scrollToRow(0);
	}

	/**
	 * Returns the given {@link ItemStack} to the stack in this inventory which's {@link Item} the given stack's item.
	 * Does nothing in config mode.
	 * @param itemStack - The item stack to return.
	 * @return {@code true} if the item stack was returned successfully, {@code false} if there was no stack in this
	 * inventory matching the given stack's item.
	 */
	public boolean returnItemStack(ItemStack itemStack) {

		// Return success in config mode. No items in the inventory should be changed.
		if (this.inventoryManager.isConfigMode() || this.hasInfiniteCodeItems) {
			return true;
		}

		// Return the item to the inventory.
		Item item = itemStack.getItem();
		for (int invIndex = 0; invIndex < this.codeItemEntries.size(); invIndex++) {
			CodeItemEntry iconAmountCouple = this.codeItemEntries.get(invIndex);
			if (this.iconRetriever.getOrderedProgrammingIcons().get(iconAmountCouple.getIconId()).equals(item)) {
				ItemStack stack = this.inventory[invIndex];
				if (stack.isEmpty()) {
					this.inventory[invIndex] = new ItemStack(item, itemStack.getCount());
				} else {
					stack.grow(itemStack.getCount());
				}
				return true;
			}
		}
		return false;
	}

	/**
	 * This function sets the Page of the ItemProgrammingIcons on the Config Screen.
	 * Then it loads the Icons for that page and preps the this.nextPage
	 * @param newRow Index of the page
	 */
	public void scrollToRow(int newRow) {
		if (this.containerHandler != null) {
			Container container = this.containerHandler.getContainerObj();
			if (container instanceof TurtleContainerSC) {
				InventorySlotManager programmingInventorySlotManager = ((TurtleContainerSC) container).getSlotManager().getProgrammingInventorySlotManager();
				programmingInventorySlotManager.scrollToRow(newRow);

			}
		}
	}

	public ProgrammingIconRetriever getProgrammingIconRetriever() {
		return this.iconRetriever;
	}

	public void setContainerHandler(GuiScreenHandler turtleGuiContainerSC) {
		this.containerHandler = turtleGuiContainerSC;
	}

	/**
	 * Counts the amount of leading adjacent non-empty items in the inventory.
	 * @return the item count.
	 */
	public int countLeadingNonEmptyInventoryItems() {
		for (int i = 0; i < this.inventory.length; i++) {
			if (this.inventory[i].isEmpty()) {
				return i;
			}
		}
		return this.inventory.length;
	}
}
