package computercraftsc.client.gui.widgets;

import computercraftsc.SCGlobal;
import computercraftsc.client.gui.widgets.WidgetButton.IButtonListener;

import static computercraftsc.client.gui.GuiConstants.SCROLL_BAR_BUTTONS_HEIGHT;

import java.util.function.Consumer;

import com.mojang.blaze3d.matrix.MatrixStack;

/**
 * Used in combination with {@link ScrollPaneWidget} to provide scrollable widget support.
 */
public class ScrollBarWidget extends Widget implements TwoLayerStateDrawable {
	
	private final WidgetImageButton scrollBarTopBtn;
	private final WidgetImageButton scrollBarBottomBtn;
	private final WidgetImage scrollBarMiddleImg;
	private final WidgetImage scrollBarSliderImg;
	private final Consumer<Float> progressUpdateCallback;
	
	private int scrollBarSliderDragOffset = -1;
	
	/**
	 * Creates a new scroll bar widget.
	 * @param xOffset - The x coordinate relative to the parent left top corner.
	 * @param yOffset - The y coordinate relative to the parent left top corner.
	 * @param width - The scroll bar width.
	 * @param height - The scroll bar height.
	 * @param progressUpdateCallback - A callback that will be called when the bar progress is updated through user
	 * input. The progress that is passed is in range (0, 1).
	 */
	public ScrollBarWidget(int xOffset, int yOffset, int width, int height, Consumer<Float> progressUpdateCallback,
			IButtonListener scrollUpBtnListener, IButtonListener scrollDownBtnListener) {
		super(xOffset, yOffset, width, height);
		this.scrollBarTopBtn = new WidgetImageButton(SCGlobal.SCROLL_BAR_TOP_TEXTURE,
				0, 0, width, SCROLL_BAR_BUTTONS_HEIGHT, scrollUpBtnListener);
		this.scrollBarTopBtn.setParent(this);
		this.scrollBarBottomBtn = new WidgetImageButton(SCGlobal.SCROLL_BAR_BOTTOM_TEXTURE,
				0, height - SCROLL_BAR_BUTTONS_HEIGHT, width, SCROLL_BAR_BUTTONS_HEIGHT,
				scrollDownBtnListener);
		this.scrollBarBottomBtn.setParent(this);
		this.scrollBarMiddleImg = new WidgetImage(SCGlobal.SCROLL_BAR_MIDDLE_TEXTURE,
				0, SCROLL_BAR_BUTTONS_HEIGHT, width, height - 2 * SCROLL_BAR_BUTTONS_HEIGHT);
		this.scrollBarMiddleImg.setParent(this);
		this.scrollBarSliderImg = new WidgetImage(SCGlobal.SCROLL_BAR_SLIDER_TEXTURE,
				0, SCROLL_BAR_BUTTONS_HEIGHT, width, height - 2 * SCROLL_BAR_BUTTONS_HEIGHT);
		this.scrollBarSliderImg.setParent(this);
		this.progressUpdateCallback = progressUpdateCallback;
	}
	
	/**
	 * Updates the scroll bar slider size and position.
	 * @param fullContentHeight - The full height of the scrollable content.
	 * @param visibleContentHeight - The visible height of the scrollable content.
	 * @param currentContentOffset - The current scrolled offset from the content top.
	 */
	public void updateBar(int fullContentHeight, int visibleContentHeight, int currentContentOffset) {
		
		// Compute scroll progress.
		float progress = (visibleContentHeight >= fullContentHeight ? 0
				: currentContentOffset / (float) (fullContentHeight - visibleContentHeight));
		if(progress > 1) {
			progress = 1;
		}
		
		// Compute slider length in the scroll bar to represent the fraction of the visible content.
		float sliderLength = (visibleContentHeight >= fullContentHeight
				? this.getHeight() - 2 * SCROLL_BAR_BUTTONS_HEIGHT
				: (this.getHeight() - 2 * SCROLL_BAR_BUTTONS_HEIGHT)
						* visibleContentHeight / (float) fullContentHeight);
		
		// Resize and position the scroll bar slider.
		this.scrollBarSliderImg.resize(this.getWidth(), (int) (sliderLength + 0.5));
		this.scrollBarSliderImg.setPosition(this.scrollBarSliderImg.getRelXPosition(), SCROLL_BAR_BUTTONS_HEIGHT
				+ (int) (progress * (this.getHeight() - 2 * SCROLL_BAR_BUTTONS_HEIGHT - sliderLength) + 0.5));
	}
	
	@Override
	public boolean mousePressed(int mouseX, int mouseY, int mouseButton) {
		boolean handled = this.scrollBarTopBtn.mousePressed(mouseX - this.scrollBarTopBtn.getRelXPosition(),
				mouseY - this.scrollBarTopBtn.getRelYPosition(), mouseButton);
		handled |= this.scrollBarBottomBtn.mousePressed(mouseX - this.scrollBarBottomBtn.getRelXPosition(),
				mouseY - this.scrollBarBottomBtn.getRelYPosition(), mouseButton);
		if(mouseButton == 0 && mouseX >= 0 && mouseX < this.getWidth()
				&& mouseY >= this.scrollBarSliderImg.getRelYPosition()
				&& mouseY < this.scrollBarSliderImg.getRelYPosition() + this.scrollBarSliderImg.getHeight()) {
			this.scrollBarSliderDragOffset = mouseY - this.scrollBarSliderImg.getRelYPosition();
			handled = true;
		}
		return handled;
	}
	
	@Override
	public boolean mouseReleased(int mouseX, int mouseY, int mouseButton) {
		boolean handled = this.scrollBarTopBtn.mouseReleased(mouseX - this.scrollBarTopBtn.getRelXPosition(),
				mouseY - this.scrollBarTopBtn.getRelYPosition(), mouseButton);
		handled |= this.scrollBarBottomBtn.mouseReleased(mouseX - this.scrollBarBottomBtn.getRelXPosition(),
				mouseY - this.scrollBarBottomBtn.getRelYPosition(), mouseButton);
		if(this.scrollBarSliderDragOffset != -1) {
			this.scrollBarSliderDragOffset = -1;
			handled = true;
		}
		return handled;
	}
	
	@Override
	public void mouseMoved(int mouseX, int mouseY) {
		this.scrollBarTopBtn.mouseMoved(mouseX - this.scrollBarTopBtn.getRelXPosition(),
				mouseY - this.scrollBarTopBtn.getRelYPosition());
		this.scrollBarBottomBtn.mouseMoved(mouseX - this.scrollBarBottomBtn.getRelXPosition(),
				mouseY - this.scrollBarBottomBtn.getRelYPosition());
		if(this.scrollBarSliderDragOffset != -1) {
			int newSliderY = mouseY - this.scrollBarSliderDragOffset;
			int sliderLength = this.scrollBarSliderImg.getHeight();
			float progress = newSliderY / (float) (this.getHeight() - 2 * SCROLL_BAR_BUTTONS_HEIGHT - sliderLength);
			progress = (progress < 0 ? 0 : (progress > 1 ? 1 : progress));
			this.scrollBarSliderImg.setPosition(this.scrollBarSliderImg.getRelXPosition(), SCROLL_BAR_BUTTONS_HEIGHT
					+ (int) (progress * (this.getHeight() - 2 * SCROLL_BAR_BUTTONS_HEIGHT - sliderLength) + 0.5));
			this.progressUpdateCallback.accept(progress);
		}
	}
	
	@Override
	public void drawGuiContainerBackgroundLayer(MatrixStack matrixStack, int mouseX, int mouseY) {
		this.scrollBarTopBtn.drawGuiContainerBackgroundLayer(matrixStack,
				mouseX - this.scrollBarTopBtn.getRelXPosition(), mouseY - this.scrollBarTopBtn.getRelYPosition());
		this.scrollBarTopBtn.drawGuiContainerForegroundLayer(matrixStack,
				mouseX - this.scrollBarTopBtn.getRelXPosition(), mouseY - this.scrollBarTopBtn.getRelYPosition());
		this.scrollBarBottomBtn.drawGuiContainerBackgroundLayer(matrixStack,
				mouseX - this.scrollBarBottomBtn.getRelXPosition(), mouseY - this.scrollBarBottomBtn.getRelYPosition());
		this.scrollBarBottomBtn.drawGuiContainerForegroundLayer(matrixStack,
				mouseX - this.scrollBarBottomBtn.getRelXPosition(), mouseY - this.scrollBarBottomBtn.getRelYPosition());
		this.scrollBarMiddleImg.drawGuiContainerBackgroundLayer(matrixStack,
				mouseX - this.scrollBarMiddleImg.getRelXPosition(), mouseY - this.scrollBarMiddleImg.getRelYPosition());
		this.scrollBarMiddleImg.drawGuiContainerForegroundLayer(matrixStack,
				mouseX - this.scrollBarMiddleImg.getRelXPosition(), mouseY - this.scrollBarMiddleImg.getRelYPosition());
	}
	
	@Override
	public void drawGuiContainerForegroundLayer(MatrixStack matrixStack, int mouseX, int mouseY) {
		this.scrollBarSliderImg.drawGuiContainerBackgroundLayer(matrixStack,
				mouseX - this.scrollBarSliderImg.getRelXPosition(), mouseY - this.scrollBarSliderImg.getRelYPosition());
		this.scrollBarSliderImg.drawGuiContainerForegroundLayer(matrixStack,
				mouseX - this.scrollBarSliderImg.getRelXPosition(), mouseY - this.scrollBarSliderImg.getRelYPosition());
	}
}
