package computercraftsc.client.gui.widgets;

import computercraftsc.SCGlobal;
import computercraftsc.client.gui.inventories.Inventory;
import computercraftsc.shared.turtle.core.code.compiler.CompileResult;
import computercraftsc.shared.turtle.core.code.compiler.TurtleLangCompiler;
import computercraftsc.shared.turtle.core.code.compiler.exception.SyntaxException;
import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.FontRenderer;
import net.minecraft.client.resources.I18n;
import net.minecraft.util.text.StringTextComponent;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import com.mojang.blaze3d.matrix.MatrixStack;

import static computercraftsc.client.gui.GuiConstants.TERMINAL_TEXT_X_OFFSET;
import static computercraftsc.client.gui.GuiConstants.TERMINAL_TEXT_Y_OFFSET;

/**
 * This {@link Widget} is used to show the Lua code representation of a {@link Program}.
 */
public class TerminalWidget extends WidgetImage implements ScrollableContent {
	
	public static final String NO_CODE_WRITTEN = "gui.computercraftsc:no_code_written";
	public static final String COMPILE_ERROR_IN_CODE = "gui.computercraftsc:compile_error_in_code";
	private static final int FONT_LINE_HEIGHT = 10;
	
	private final Inventory codeInventory;
	private List<String> codeLines = Collections.emptyList();
	private int scrollRowOffset = 0;
	
	/**
	 * Creates a new {@link TerminalWidget} that will contain the string representation of the program in the
	 * given {@link Inventory}.
	 * @param xOffset
	 * @param yOffset
	 * @param width
	 * @param height
	 * @param codeInventory - The inventory containing the code.
	 */
	public TerminalWidget(int xOffset, int yOffset, int width, int height, Inventory codeInventory) {
		super(SCGlobal.TERMINAL_WIDGET_TEXTURE, xOffset, yOffset, width, height);
		this.codeInventory = codeInventory;
	}
	
	@Override
	public void setVisible(boolean visible) {
		super.setVisible(visible);
		if(visible) {
			this.updateCode();
			
			// Scroll up if visible content was removed.
			if(this.codeLines == null) {
				this.scrollRowOffset = 0;
			} else {
				int maxNumCodeLines = this.getHeight() / FONT_LINE_HEIGHT;
				if(maxNumCodeLines + this.scrollRowOffset > this.codeLines.size()) {
					this.scrollRowOffset = Math.max(0, this.codeLines.size() - maxNumCodeLines);
				}
			}
		}
	}
	
	/**
	 * Updates the program in this {@link TerminalWidget} from the code inventory.
	 */
	private void updateCode() {
		try {
			CompileResult result = TurtleLangCompiler.compile(this.codeInventory.getInventory());
			String codeString = result.getAST().toCodeString();
			this.codeLines = (codeString.isEmpty()
					? Collections.emptyList() : Arrays.asList(codeString.replace("\t", "    ").split("\n")));
		} catch (SyntaxException e) {
			this.codeLines = null;
		}
	}
	
	@Override
	public void drawGuiContainerForegroundLayer(MatrixStack matrixStack, int mouseX, int mouseY) {
		super.drawGuiContainerForegroundLayer(matrixStack, mouseX, mouseY);
		
		@SuppressWarnings("resource")
		FontRenderer fontRenderer = Minecraft.getInstance().fontRenderer;
		int x = this.getAbsoluteXPosition() + TERMINAL_TEXT_X_OFFSET;
		int y = this.getAbsoluteYPosition() + TERMINAL_TEXT_Y_OFFSET;
		
		if(this.codeLines == null) {
			fontRenderer.drawText(matrixStack, new StringTextComponent(I18n.format(COMPILE_ERROR_IN_CODE)),
					x, y, SCGlobal.GUI_COLOR_WHITE);
		} else if(this.codeLines.isEmpty()) {
			fontRenderer.drawText(matrixStack, new StringTextComponent(I18n.format(NO_CODE_WRITTEN)),
					x, y, SCGlobal.GUI_COLOR_WHITE);
		} else {
			int numOnScreenRows = Math.min(
					this.getHeight() / FONT_LINE_HEIGHT, this.codeLines.size() - this.scrollRowOffset);
			for(int i = 0; i < numOnScreenRows; i++) {
				int codeLinesInd = this.scrollRowOffset + i;
				fontRenderer.drawText(matrixStack,
						new StringTextComponent((codeLinesInd + 1) + ". " + this.codeLines.get(codeLinesInd)),
						x, y, SCGlobal.GUI_COLOR_WHITE);
				y += FONT_LINE_HEIGHT;
			}
		}
	}
	
	@Override
	public int getContentHeight() {
		return (this.codeLines == null ? 0 : this.codeLines.size()) * FONT_LINE_HEIGHT;
	}
	
	@Override
	public int getVisibleContentHeight() {
		return (this.getHeight() / FONT_LINE_HEIGHT) * FONT_LINE_HEIGHT;
	}
	
	@Override
	public int getScrollHeightStep() {
		return FONT_LINE_HEIGHT;
	}
	
	@Override
	public void onScrollTopOffsetChanged(int topOffset) {
		this.scrollRowOffset = Math.max(0, topOffset / this.getScrollHeightStep());
	}
	
	@Override
	public int getScrollTopOffset() {
		return -1;
	}
}
