package computercraftsc.client.gui.widgets;

import java.util.ArrayList;
import java.util.Iterator;

import computercraftsc.client.gui.KeyCodes;
import net.minecraft.block.Block;
import net.minecraft.util.registry.Registry;
import net.minecraft.util.text.TranslationTextComponent;

public class WidgetBlockSelectionPopup extends WidgetPopup {

	public static final int MATCHING_BLOCKS_LIMIT = 128;

	private IBlockSelectionListener m_listener;
	private WidgetTextBox m_textBox;
	private WidgetListBox m_listBox;
	private ArrayList<Block> m_matchingBlocks;

	public WidgetBlockSelectionPopup(int x, int y, Block defaultBlock, IBlockSelectionListener listener) {
		super(x, y, 180, 18);
		this.m_listener = listener;
		this.m_matchingBlocks = new ArrayList<Block>();
		WidgetContainer content = getContent();
		final WidgetButton okButton = new WidgetTextButton("gui.computercraftsc:popup_ok", content.getWidth() - 24,
				0, 24, (WidgetButton button) -> this.tryConfirm());
		content.addWidget(okButton);
		this.m_textBox = new WidgetTextBox(getBlockName(defaultBlock), 32, TextAlignment.LEFT, 0,
				0, content.getWidth() - 24 - 2, new WidgetTextBox.ITextBoxListener() {

					@Override
					public void onTextChanged(WidgetTextBox widget) {
						refresh(widget);
					}

					@Override
					public void onFocusChanged(WidgetTextBox widget) {
						refresh(widget);
					}

					@Override
					public boolean isTextValid(String text) {
						return true;
					}

					private void refresh(WidgetTextBox widget) {
						if(widget.getFocus()) {
							WidgetBlockSelectionPopup.this.findMatchingBlocks(true);
							okButton.setLocked((WidgetBlockSelectionPopup.this.m_listBox.getSelection() < 0));
						}
					}
				});
		content.addWidget(this.m_textBox);
		this.m_listBox = new WidgetListBox(0, this.m_textBox.getRelYPosition() + this.m_textBox.getHeight() + 2,
				content.getWidth(), 7, new String[0], new IListBoxListener() {

					@Override
					public void onSelectionChanged(WidgetListBox widget) {
						refresh(widget);
					}

					@Override
					public void onFocusChanged(WidgetListBox widget) {
						refresh(widget);
					}

					private void refresh(WidgetListBox widget) {
						okButton.setLocked((widget.getSelection() < 0));
						if (!WidgetBlockSelectionPopup.this.m_textBox.getFocus()) {
							String selectedText = widget.getSelectedText();
							if (selectedText != null)
								WidgetBlockSelectionPopup.this.m_textBox.setText(selectedText);
						}
					}
				}, true);
		content.addWidget(this.m_listBox);
		this.resizeContent(this.m_listBox.getWidth(), this.m_listBox.getRelYPosition() + this.m_listBox.getHeight());
		this.findMatchingBlocks(defaultBlock != null);
		if(defaultBlock != null) {
			for(int i = 0; i < this.m_matchingBlocks.size(); i++) {
				if(this.m_matchingBlocks.get(i) == defaultBlock) {
					this.m_listBox.setSelection(i);
					break;
				}
			}
		}
		okButton.setLocked((this.m_listBox.getSelection() < 0));
		this.m_textBox.setFocus(true);
	}

	@Override
	public boolean keyPressed(int keyCode, int scanCode, int modifiers) {
		if(keyCode == KeyCodes.TAB) {
			if(this.m_textBox.getFocus()) {
				this.m_textBox.setFocus(false);
				this.m_listBox.setFocus(true);
				if(this.m_listBox.getSelection() < 0 && this.m_matchingBlocks.size() > 0) {
					this.m_listBox.setSelection(0);
					this.m_textBox.setText(this.m_listBox.getSelectedText());
				}
			} else if(this.m_listBox.getFocus()) {
				this.m_listBox.setFocus(false);
				this.m_textBox.setFocus(true);
			}
		}
		return super.keyPressed(keyCode, scanCode, modifiers);
	}

	@Override
	protected void onCancel() {
		this.m_listener.onCancel(this);
	}

	@Override
	protected void onConfirm() {
		if(tryConfirm()) {
			playClickSound();
		}
	}

	private boolean tryConfirm() {
		Block selectedBlock = getSelectedBlock();
		if(selectedBlock != null) {
			this.m_listener.onConfirmation(this, selectedBlock);
			return true;
		}
		return false;
	}

	private Block getSelectedBlock() {
		int selection = this.m_listBox.getSelection();
		if(selection >= 0) {
			return this.m_matchingBlocks.get(selection);
		}
		return null;
	}

	private String getBlockName(Block block) {
		if(block != null) {
			return new TranslationTextComponent(block.getTranslationKey()).getString();
		}
		return "";
	}

	private void findMatchingBlocks(boolean selectFirstMatch) {
		this.m_matchingBlocks.clear();
		int match = -1;
		int i = 0;
		String filter = this.m_textBox.getText().toLowerCase();
		@SuppressWarnings("deprecation")
		Iterator<Block> iterator = Registry.BLOCK.iterator();
		while(iterator.hasNext()) {
			Block block = iterator.next();
			if(block != null) {
				String name = getBlockName(block).toLowerCase();
				if(name.contains(filter)) {
					this.m_matchingBlocks.add(block);
					if(name.equals(filter)) {
						match = i;
					}
					i++;
					if(this.m_matchingBlocks.size() >= MATCHING_BLOCKS_LIMIT) {
						break;
					}
				}
			}
		}
		if(match < 0 && this.m_matchingBlocks.size() > 0) {
			match = 0;
		}
		String[] names = new String[this.m_matchingBlocks.size()];
		for(int j = 0; j < names.length; j++) {
			Block block = this.m_matchingBlocks.get(j);
			names[j] = getBlockName(block);
		}
		this.m_listBox.setOptions(names);
		if(selectFirstMatch) {
			this.m_listBox.setSelection(match);
		}
	}
}
