package computercraftsc.shared.turtle.block;

import java.lang.ref.WeakReference;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import computercraftsc.ComputerCraftSC;
import computercraftsc.client.gui.ProgrammingIconRetriever;
import computercraftsc.client.gui.container.TurtleContainerSC;
import computercraftsc.client.gui.inventories.Inventory;
import computercraftsc.client.gui.inventories.InventoryManager;
import computercraftsc.client.gui.inventories.config.CodeItemEntry;
import computercraftsc.shared.RegistrySC;
import computercraftsc.shared.items.ItemTurtleSC;
import computercraftsc.shared.turtle.core.TurtleBrainSC;
import computercraftsc.shared.turtle.core.code.ProgramState;
import computercraftsc.shared.turtle.core.code.TurtleExecutor;
import dan200.computercraft.api.turtle.ITurtleUpgrade;
import dan200.computercraft.api.turtle.TurtleSide;
import dan200.computercraft.shared.computer.core.ComputerFamily;
import dan200.computercraft.shared.turtle.blocks.TileTurtle;
import dan200.computercraft.shared.turtle.core.InteractDirection;
import dan200.computercraft.shared.util.RedstoneUtil;
import net.minecraft.block.BlockState;
import net.minecraft.entity.player.PlayerEntity;
import net.minecraft.entity.player.PlayerInventory;
import net.minecraft.entity.player.ServerPlayerEntity;
import net.minecraft.inventory.container.Container;
import net.minecraft.item.ItemStack;
import net.minecraft.item.Items;
import net.minecraft.nbt.CompoundNBT;
import net.minecraft.nbt.ListNBT;
import net.minecraft.tileentity.TileEntityType;
import net.minecraft.util.ActionResultType;
import net.minecraft.util.Direction;
import net.minecraft.util.Hand;
import net.minecraft.util.ResourceLocation;
import net.minecraft.util.Util;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.BlockRayTraceResult;
import net.minecraft.util.text.IFormattableTextComponent;
import net.minecraft.util.text.ITextComponent;
import net.minecraft.util.text.StringTextComponent;
import net.minecraft.util.text.TextFormatting;
import net.minecraft.util.text.TranslationTextComponent;
import net.minecraftforge.fml.network.NetworkHooks;

public class TileTurtleSC extends TileTurtle {

	private String customName = null;
	private final ProgrammingIconRetriever programmingIconRetriever = new ProgrammingIconRetriever();
	private InventoryManager inventoryManager = new InventoryManager(this.programmingIconRetriever);
	private int[] redstoneOutputPower = new int[] {0, 0, 0, 0, 0, 0};
	private boolean interactionBlocked = false; // Whether or not to allow players to interact with the turtle.
	private boolean inventoriesLocked = false; // Whether or not to allow changing items in the inventories.
	private boolean turtleInventoryLocked = false; // Whether or not to allow changing items in the turtle inventory.
	private WeakReference<PlayerEntity> lastUsingPlayer = new WeakReference<>(null); // Last player that opened the GUI.
	private TurtleExecutor turtleExecutor = (ComputerCraftSC.sideRunsServer() ? new TurtleExecutor(this) : null);

	public TileTurtleSC(TileEntityType<TileTurtleSC> tileEntityType) {
		super(tileEntityType, ComputerFamily.NORMAL);

		// Overwrite turtle brain in super class.
		try {
			Field brainField = TileTurtle.class.getDeclaredField("brain");
			brainField.setAccessible(true);
			brainField.set(this, new TurtleBrainSC(this));
		} catch (IllegalArgumentException | IllegalAccessException | NoSuchFieldException | SecurityException e) {
			throw new Error(e);
		}

		// Set inventoryChanged in super class to prevent out-of-bounds access to previousInventory.
		// Note that for this to work, it is required that we do not call tick() on the super class.
		try {
			Field inventoryChangedField = TileTurtle.class.getDeclaredField("inventoryChanged");
			inventoryChangedField.setAccessible(true);
			inventoryChangedField.set(this, true);
		} catch (IllegalArgumentException | IllegalAccessException | NoSuchFieldException | SecurityException e) {
			throw new Error(e);
		}
	}

	private void setLastUsingPlayer(PlayerEntity player) {
		this.lastUsingPlayer = new WeakReference<>(player);
		this.getInventoryManager().setInventoryPlayer(player.inventory);
	}

	public PlayerEntity getLastUsingPlayer() {
		return this.lastUsingPlayer.get();
	}

	public InventoryManager getInventoryManager() {
		return this.inventoryManager;
	}

	public TurtleExecutor getExecutor() {
		return this.turtleExecutor;
	}

	@Override
	public void openInventory(PlayerEntity player) {
		System.out.println("[DEBUG] openInventory method called.");
	}

	@Override
	public ActionResultType onActivate(PlayerEntity player, Hand hand, BlockRayTraceResult hit) {

		// Ignore client-side GUI open requests.
		if(this.world.isRemote) {
			return ActionResultType.SUCCESS;
		}

		// Ignore the GUI open request if turtle interaction is blocked.
		if(this.interactionBlocked) {
			return ActionResultType.SUCCESS;
		}

		// Ignore the GUI open request if a GUI for this turtle is already open for another player.
		PlayerEntity lastUsingPlayer = this.getLastUsingPlayer();
		if(lastUsingPlayer != null && lastUsingPlayer.openContainer instanceof TurtleContainerSC
				&& ((TurtleContainerSC) lastUsingPlayer.openContainer).getTileTurtleSC() == this
				&& !lastUsingPlayer.equals(player) && !((ServerPlayerEntity) lastUsingPlayer).hasDisconnected()) {
			player.sendMessage(new TranslationTextComponent(
					"message.computercraftsc.turtle_already_opened_by_other_player")
					.mergeStyle(TextFormatting.RED), Util.DUMMY_UUID);
			return ActionResultType.SUCCESS;
		}

		// Get whether the inventory should be opened in config mode.
		boolean isConfigMode = this.isConfigMode(player);

		// Initialize inventory manager.
		this.inventoryManager.onGUIOpen(player, isConfigMode);

		// Set last using player.
		this.setLastUsingPlayer(player);

		// Send GUI open request to the client.
		List<CodeItemEntry> codeItemEntries = this.inventoryManager.getProgrammingInventory().getCodeItemEntries();
		Set<Integer> lockedCodeInvSlotIds = this.inventoryManager.getCodingInventory().getLockedInventorySlotIds();
		NetworkHooks.openGui((ServerPlayerEntity) player, this, (buf) -> {
			buf.writeBlockPos(this.pos);
			buf.writeBoolean(isConfigMode);
			buf.writeBoolean(this.inventoriesLocked);
			buf.writeBoolean(this.turtleInventoryLocked);
			buf.writeInt(codeItemEntries.size());
			for(int i = 0; i < codeItemEntries.size(); i++) {
				buf.writeCompoundTag(codeItemEntries.get(i).createCompoundNBT());
			}
			buf.writeBoolean(this.inventoryManager.getProgrammingInventory().getHasInfiniteCodeItems());
			buf.writeInt(lockedCodeInvSlotIds.size());
			for(int lockedCodeInventorySlotId : lockedCodeInvSlotIds) {
				buf.writeInt(lockedCodeInventorySlotId);
			}
		});

		// Return success.
		return ActionResultType.SUCCESS;
	}

	@Override
	public Container createMenu(int windowId, PlayerInventory playerInv, PlayerEntity player) {
		return new TurtleContainerSC(windowId, playerInv, this);
	}

	@Override
	public void closeInventory(PlayerEntity player) {
		if(!player.world.isRemote && this.inventoryManager.isConfigMode()) {
			this.saveNewConfig();
			this.inventoryManager.getCodingInventory().clear();
		}
	}

	private void saveNewConfig() {
		List<CodeItemEntry> newCodeItemEntries = this.inventoryManager.getNewConfigItems();
		this.inventoryManager.setCodeItemEntries(newCodeItemEntries);
		this.markDirty();
	}

	@Override
	public int getSizeInventory() {
		return this.inventoryManager.getSizeInventory();
	}

	@Override
	public boolean isEmpty() {
		return this.inventoryManager.isEmpty();
	}

	@Override
	public ItemStack getStackInSlot(int index) {
		return this.inventoryManager.getStackInSlot(index);
	}

	@Override
	public ItemStack removeStackFromSlot(int index) {
		return this.inventoryManager.removeStackFromSlot(index);
	}

	@Override
	public ItemStack decrStackSize(int index, int count) {
		this.markDirty();
		return this.inventoryManager.decrStackSize(index, count);
	}

	@Override
	public void setInventorySlotContents(int index, ItemStack stack) {
		this.inventoryManager.setInventorySlotContents(index, stack);
		this.markDirty();
	}

	@Override
	public boolean isItemValidForSlot(int index, ItemStack stack) {
		return true;
	}

	public boolean hasMoved() {

		// Call private super.hasMoved().
		try {
			Method hasMovedMethod = TileTurtle.class.getDeclaredMethod("hasMoved");
			hasMovedMethod.setAccessible(true);
			return (boolean) hasMovedMethod.invoke(this);
		} catch (IllegalAccessException | IllegalArgumentException
				| InvocationTargetException | NoSuchMethodException | SecurityException e) {
			throw new Error(e);
		}
	}

	@Override
	public void destroy() {

		// Update redstone in all directions.
		for(Direction dir : Direction.values()) {
			RedstoneUtil.propagateRedstoneOutput(this.world, this.getPos(), dir);
		}

		// Run turtle destroy logic (VS turtle move logic) if the turtle hasn't moved.
		if(!this.hasMoved()) {

			// Remove the server computer from the server computer registry.
			this.unload();
		}
	}

	public int getRedstoneOutput(Direction side) {
		return this.redstoneOutputPower[side.getIndex()];
	}

	public void setRedstoneOutput(Direction side, int power) {
		if(power >= 0 && power < 16 && this.redstoneOutputPower[side.getIndex()] != power) {
			this.redstoneOutputPower[side.getIndex()] = power;
			RedstoneUtil.propagateRedstoneOutput(this.getWorld(), this.getPos(), side);
		}
	}

	public void setRedstoneOutput(InteractDirection direction, int power) {
		this.setRedstoneOutput(direction.toWorldDir(this.getAccess()), power);
	}

	public int getRedstoneInput(Direction side) {
		return this.world.getRedstonePower(this.pos.offset(side), side);
	}

	public int getRedstoneInput(InteractDirection direction) {
		return this.getRedstoneInput(direction.toWorldDir(this.getAccess()));
	}

	/**
	 * Sets whether turtle interaction should be blocked (i.e. interaction with the turtle block is prevented).
	 * @param interactionBlocked - {@code true} to block interaction, {@code false} to allow interaction.
	 */
	public void setInteractionBlocked(boolean interactionBlocked) {
		this.interactionBlocked = interactionBlocked;
	}

	/**
	 * Gets whether turtle is blocked (i.e. interaction with the turtle block is prevented).
	 * @return The interaction block state.
	 */
	public boolean getInteractionBlocked() {
		return this.interactionBlocked;
	}

	/**
	 * Sets whether the inventories should be put into read-only mode.
	 * @param lock - {@code true} to lock, {@code false} to unlock.
	 */
	public void setInventoriesLocked(boolean lock) {
		this.inventoriesLocked = lock;
	}

	/**
	 * Gets whether the turtle inventories are locked into read-only mode.
	 * @return The lock state.
	 */
	public boolean getInventoriesLocked() {
		return this.inventoriesLocked;
	}

	/**
	 * Sets whether the turtle item inventory should be put into read-only mode.
	 * @param lock - {@code true} to lock, {@code false} to unlock.
	 */
	public void setTurtleInventoryLocked(boolean lock) {
		this.turtleInventoryLocked = lock;
	}

	/**
	 * Gets whether the turtle item inventory is locked into read-only mode.
	 * @return The lock state.
	 */
	public boolean getTurtleInventoryLocked() {
		return this.turtleInventoryLocked;
	}

	/**
	 * Gets this turtle as a {@link ItemTurtleSC} in an {@link ItemStack}.
	 * @return The {@link ItemStack} representing this turtle.
	 */
	public ItemStack getTurtleItem() {
		ItemTurtleSC turtleItem = RegistrySC.ModItems.TURTLE_SC.get();
		ITurtleUpgrade leftUpgrade = this.getAccess().getUpgrade(TurtleSide.LEFT);
		ITurtleUpgrade rightUpgrade = this.getAccess().getUpgrade(TurtleSide.RIGHT);
		int fuelLevel = 0;
		return turtleItem.create(-1, null, this.getColour(), leftUpgrade, rightUpgrade, fuelLevel, this.getOverlay());
	}

	private boolean isConfigMode(PlayerEntity player) {
		boolean pIsOp = player.getServer().getPlayerList().getOppedPlayers().getEntry(player.getGameProfile()) != null;
		return (pIsOp || ComputerCraftSC.isSinglePlayer()) && player.getHeldItemMainhand().getItem() == Items.REDSTONE;
	}

	@Override
	public ITextComponent getName() {
		return new StringTextComponent(this.getClass().getSimpleName());
	}

	@Override
	public boolean hasCustomName() {
		return this.customName != null && !"".equals(this.customName);
	}

	@Override
	public ITextComponent getCustomName() {
		return (this.hasCustomName() ? new StringTextComponent(this.customName) : this.getName());
	}

	@Override
	public ITextComponent getDisplayName() {
		return this.getCustomName();
	}

	@Override
	public CompoundNBT write(CompoundNBT nbt) {

		// Write TileEntity data (avoid calling the super implementation).
		ResourceLocation resourcelocation = TileEntityType.getId(this.getType());
		if(resourcelocation == null) {
			throw new RuntimeException(this.getClass() + " is missing a mapping! This is a bug!");
		} else {
			nbt.putString("id", resourcelocation.toString());
			nbt.putInt("x", this.pos.getX());
			nbt.putInt("y", this.pos.getY());
			nbt.putInt("z", this.pos.getZ());
			if(this.getTileData().size() != 0) {
				nbt.put("ForgeData", this.getTileData());
			}
			if(this.getCapabilities() != null) {
				nbt.put("ForgeCaps", serializeCaps());
			}
		}

		// Write turtle brain data.
		((TurtleBrainSC) this.getAccess()).writeToNBT(nbt);

		// Write inventories.
		writeInventoryToNBT("TurtleInventory", this.inventoryManager.getTurtleInventory(), nbt);
		writeInventoryToNBT("CodingInventory", this.inventoryManager.getCodingInventory(), nbt);

		// Write available code items.
		ListNBT codeItemEntriesNBT = new ListNBT();
		for (CodeItemEntry codeItemEntry : this.inventoryManager.getProgrammingInventory().getCodeItemEntries()) {
			codeItemEntriesNBT.add(codeItemEntry.createCompoundNBT());
		}
		nbt.put("AvailableCodeItems", codeItemEntriesNBT);

		// Write locked inventory slot ids.
		Set<Integer> lockedInvSlotIds = this.inventoryManager.getCodingInventory().getLockedInventorySlotIds();
		List<Integer> lockedInvSlotIdsList = new ArrayList<>(lockedInvSlotIds);
		lockedInvSlotIdsList.sort((i1, i2) -> Integer.compare(i1, i2));
		if(!lockedInvSlotIds.isEmpty()) {
			nbt.putIntArray("LockedCodeInvSlotIds", lockedInvSlotIdsList);
		}

		// Write infinite code items setting.
		if (this.inventoryManager.getProgrammingInventory().getHasInfiniteCodeItems()) {
			nbt.putBoolean("HasInfiniteCodeItems", true);
		}

		// Write redstone output power.
		nbt.putIntArray("RedstoneOutputPower", this.redstoneOutputPower);

		// Write turtle lock settings.
		if (this.interactionBlocked) {
			nbt.putBoolean("InteractionBlocked", true);
		}

		// Write inventory lock settings.
		if (this.inventoriesLocked) {
			nbt.putBoolean("InventoriesLocked", true);
		}
		if (this.turtleInventoryLocked) {
			nbt.putBoolean("TurtleInventoryLocked", true);
		}

		// Write execution state.
		if(this.turtleExecutor != null) {
			this.turtleExecutor.writeToNBT(nbt);
		}

		// Write optional custom name.
		if (this.hasCustomName()) {
			nbt.putString("CustomName", this.customName);
		}

		return nbt;
	}

	private static void writeInventoryToNBT(String tagName, Inventory inv, CompoundNBT nbt) {
		ListNBT list = new ListNBT();
		for (int i = 0; i < inv.getSizeInventory(); ++i) {
			ItemStack stack = inv.getStackInSlot(i);
			if(!stack.isEmpty()) {
				CompoundNBT stackTag = new CompoundNBT();
				stackTag.putByte("Slot", (byte) i);
				stack.write(stackTag);
				list.add(stackTag);
			}
		}
		nbt.put(tagName, list);
	}

	@Override
	public void read(BlockState blockState, CompoundNBT nbt) {
		
		// Read TileEntity data (avoid calling the super implementation).
		this.pos = new BlockPos(nbt.getInt("x"), nbt.getInt("y"), nbt.getInt("z"));
		if(nbt.contains("ForgeData")) {
			CompoundNBT newCustomTileData = nbt.getCompound("ForgeData");
			CompoundNBT customTileData = this.getTileData();
			for(String key : customTileData.keySet()) {
				customTileData.remove(key);
			}
			for(String key : newCustomTileData.keySet()) {
				customTileData.put(key, newCustomTileData.get(key));
			}
		}
		if(this.getCapabilities() != null && nbt.contains("ForgeCaps")) {
			this.deserializeCaps(nbt.getCompound("ForgeCaps"));
		}

		// Read turtle brain data.
		((TurtleBrainSC) this.getAccess()).readFromNBT(nbt);

		// Read inventories.
		readInventoryFromNBT("TurtleInventory", this.inventoryManager.getTurtleInventory(), nbt);
		readInventoryFromNBT("CodingInventory", this.inventoryManager.getCodingInventory(), nbt);
		this.inventoryManager.getCodingInventory().compileCode();

		// Read available code items.
		List<CodeItemEntry> codeItemEntries = new ArrayList<>();
		ListNBT config = nbt.getList("AvailableCodeItems", 10);
		for (int i = 0; i < config.size(); ++i) {
			CompoundNBT configTag = config.getCompound(i);
			codeItemEntries.add(CodeItemEntry.readFromNBT(configTag));
		}
		Collections.sort(codeItemEntries); // Ensure that the available code items are in the right order.
		this.inventoryManager.setCodeItemEntries(codeItemEntries);

		// Read locked inventory slot ids.
		int[] lockedCodeInvSlotIdsArray = nbt.getIntArray("LockedCodeInvSlotIds");
		Set<Integer> lockedCodeInvSlotIds = this.inventoryManager.getCodingInventory().getLockedInventorySlotIds();
		for(int lockedCodeInvSlotId : lockedCodeInvSlotIdsArray) {
			lockedCodeInvSlotIds.add(lockedCodeInvSlotId);
		}

		// Read infinite code items setting.
		this.inventoryManager.getProgrammingInventory().setHasInfiniteCodeItems(nbt.getBoolean("HasInfiniteCodeItems"));

		// Read redstone output power.
		this.redstoneOutputPower = (nbt.contains("RedstoneOutputPower", 11)
				? nbt.getIntArray("RedstoneOutputPower") : new int[] {0, 0, 0, 0, 0, 0});

		// Read turtle lock setting.
		this.interactionBlocked = nbt.getBoolean("InteractionBlocked"); // Defaults to false.

		// Read inventory lock settings.
		this.inventoriesLocked = nbt.getBoolean("InventoriesLocked"); // Defaults to false.
		this.turtleInventoryLocked = nbt.getBoolean("TurtleInventoryLocked"); // Defaults to false.

		// Read execution state.
		if(this.turtleExecutor != null) {
			this.turtleExecutor.readFromNBT(nbt);
		}

		// Read optional custom name.
		if (nbt.contains("CustomName", 8)) {
			this.customName = nbt.getString("CustomName");
		}
	}

	private static void readInventoryFromNBT(String tagName, Inventory inv, CompoundNBT nbt) {

		// Return if this inventory isn't stored for this turtle.
		if (!nbt.contains(tagName, 9)) {
			return;
		}

		// Read the inventory data into the inventory.
		ListNBT list = nbt.getList(tagName, 10);
		for (int i = 0; i < list.size(); ++i) {
			CompoundNBT stackTag = list.getCompound(i);
			int slot = stackTag.getByte("Slot") & 255;
			ItemStack stack = ItemStack.read(stackTag);
			inv.setInventorySlotContents(slot, stack);
		}
	}

	@Override
	public void transferStateFrom(TileTurtle copy) {
		super.transferStateFrom(copy);
		TileTurtleSC tileTurtleSC = (TileTurtleSC) copy;
		this.customName = tileTurtleSC.customName;
		this.inventoryManager = tileTurtleSC.inventoryManager;
		this.redstoneOutputPower = tileTurtleSC.redstoneOutputPower;
		this.interactionBlocked = tileTurtleSC.interactionBlocked;
		this.inventoriesLocked = tileTurtleSC.inventoriesLocked;
		this.turtleInventoryLocked = tileTurtleSC.turtleInventoryLocked;
		this.lastUsingPlayer = tileTurtleSC.lastUsingPlayer;
		this.turtleExecutor = tileTurtleSC.turtleExecutor;
		if(this.turtleExecutor != null) {
			this.turtleExecutor.setTileTurtle(this);
		}
	}

	/**
	 * Starts the turtle program if the turtle is currently not running a program.
	 * Closes the turtle inventory screen for the last using player in the process.
	 */
	private void runProgram() {
		if (this.turtleExecutor.getState() != ProgramState.RUNNING) {
			this.turtleExecutor.start();
		}

		PlayerEntity lastUsingPlayer = this.getLastUsingPlayer();
		if(lastUsingPlayer != null) {
			lastUsingPlayer.closeScreen();
		}
	}

	@Override
	public void tick() {
		if(this.turtleExecutor != null) {
			this.turtleExecutor.tick();
		}
		((TurtleBrainSC) this.getAccess()).update();
	}

	/**
	 * Called by {@link TurtleBrainSC} when the program state changes.
	 * @param oldState     - The old program state.
	 * @param newState     - The new program state.
	 * @param errorMessage - An optional error message or {@code null} if none.
	 */
	public void onProgramStateChanged(
			ProgramState oldState, ProgramState newState, IFormattableTextComponent errorMessage) {

		// Send error message to the executing player when possible.
		PlayerEntity lastUsingPlayer = this.getLastUsingPlayer();
		if(newState == ProgramState.ERRORED && lastUsingPlayer != null) {
			lastUsingPlayer.sendMessage(new StringTextComponent(TextFormatting.GOLD + "["
					+ TextFormatting.DARK_AQUA + "TurtleSC" + TextFormatting.GOLD + "] "
					+ TextFormatting.RED).appendSibling((errorMessage != null ? errorMessage
							: new StringTextComponent("null")).mergeStyle(TextFormatting.RED)), Util.DUMMY_UUID);
		}
	}

	/**
	 * If the playbutton is pressed while not in configMode, run the program.
	 */
	public void handlePlayButtonPress() {
		if (!this.world.isRemote) {
			if (!this.inventoryManager.isConfigMode()) {
				if (!this.inventoryManager.getCodingInventory().codeHasCompileErrors()) {
					this.runProgram();
				}
			}
		}
	}
}
