package computercraftsc.shared.turtle.core.code;

import computercraftsc.event.EventManager;
import computercraftsc.event.ProgramStateChangedEvent;
import computercraftsc.shared.turtle.block.TileTurtleSC;
import computercraftsc.shared.turtle.core.code.ast.AssignStatement;
import computercraftsc.shared.turtle.core.code.ast.BinaryExpression;
import computercraftsc.shared.turtle.core.code.ast.BinaryOperator;
import computercraftsc.shared.turtle.core.code.ast.BooleanConstant;
import computercraftsc.shared.turtle.core.code.ast.BreakStatement;
import computercraftsc.shared.turtle.core.code.ast.CommentStatement;
import computercraftsc.shared.turtle.core.code.ast.Expression;
import computercraftsc.shared.turtle.core.code.ast.ForStatement;
import computercraftsc.shared.turtle.core.code.ast.FunctionCallExpression;
import computercraftsc.shared.turtle.core.code.ast.FunctionCallStatement;
import computercraftsc.shared.turtle.core.code.ast.IfStatement;
import computercraftsc.shared.turtle.core.code.ast.IntegerConstant;
import computercraftsc.shared.turtle.core.code.ast.JumptoStatement;
import computercraftsc.shared.turtle.core.code.ast.Program;
import computercraftsc.shared.turtle.core.code.ast.RepeatStatement;
import computercraftsc.shared.turtle.core.code.ast.Statement;
import computercraftsc.shared.turtle.core.code.ast.Statements;
import computercraftsc.shared.turtle.core.code.ast.StringConstant;
import computercraftsc.shared.turtle.core.code.ast.UnaryExpression;
import computercraftsc.shared.turtle.core.code.ast.UnaryOperator;
import computercraftsc.shared.turtle.core.code.ast.Variable;
import computercraftsc.shared.turtle.core.code.ast.WhileStatement;
import computercraftsc.shared.turtle.core.code.compiler.CompileResult;
import computercraftsc.shared.turtle.core.code.compiler.TurtleLangCompiler;
import computercraftsc.shared.turtle.core.code.compiler.exception.SyntaxException;
import dan200.computercraft.api.turtle.ITurtleCommand;
import dan200.computercraft.api.turtle.TurtleCommandResult;
import dan200.computercraft.api.turtle.TurtleSide;
import dan200.computercraft.shared.turtle.core.InteractDirection;
import dan200.computercraft.shared.turtle.core.MoveDirection;
import dan200.computercraft.shared.turtle.core.TurnDirection;
import dan200.computercraft.shared.turtle.core.TurtleCompareCommand;
import dan200.computercraft.shared.turtle.core.TurtleDetectCommand;
import dan200.computercraft.shared.turtle.core.TurtleDropCommand;
import dan200.computercraft.shared.turtle.core.TurtleInspectCommand;
import dan200.computercraft.shared.turtle.core.TurtleMoveCommand;
import dan200.computercraft.shared.turtle.core.TurtlePlaceCommand;
import dan200.computercraft.shared.turtle.core.TurtleSuckCommand;
import dan200.computercraft.shared.turtle.core.TurtleToolCommand;
import dan200.computercraft.shared.turtle.core.TurtleTurnCommand;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.item.Items;
import net.minecraft.nbt.ByteNBT;
import net.minecraft.nbt.CompoundNBT;
import net.minecraft.nbt.INBT;
import net.minecraft.nbt.IntNBT;
import net.minecraft.nbt.StringNBT;
import net.minecraft.tileentity.ITickableTileEntity;
import net.minecraft.util.Util;
import net.minecraft.util.text.ChatType;
import net.minecraft.util.text.IFormattableTextComponent;
import net.minecraft.util.text.StringTextComponent;
import net.minecraft.util.text.TranslationTextComponent;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.util.Random;

public class TurtleExecutor implements ITickableTileEntity {

    private static final Logger LOGGER = LogManager.getLogger();

	private static final int TICKS_BEFORE_START = 2;
	private static final int TICKS_PER_ACTION = 8;

	private TileTurtleSC tile;
	private ProgramState state = ProgramState.STOPPED;
	private int tickCount = 0;
	private int actionDurationTicks = TICKS_PER_ACTION;
	private int programCounter = 0;
	private List<Statement> instructions = null;
	private Map<String, Object> variableMap = new HashMap<>();

	/**
	 * The maximum number of instructions that can be executed in a turtle each tick. Statements with subexpressions
	 * count as a single instruction in this context.
	 */
	private static final int MAX_INSTRUCTIONS_PER_TICK = 1000;

	public TurtleExecutor(TileTurtleSC tile) {
		this.tile = tile;
	}

	public void setTileTurtle(TileTurtleSC tile) {
		this.tile = tile;
	}

	/**
	 * Reads the execution state from NBT and continues execution if the state is {@link ProgramState#RUNNING}.
	 * Note that this should be called after the coding inventory items have been loaded, as this depends on that code.
	 * @param nbt - The NBT data.
	 */
	public void readFromNBT(CompoundNBT nbt) {
		if(nbt.contains("ExecutionState", 10)) {
			CompoundNBT exeComp = nbt.getCompound("ExecutionState");
			ProgramState state = ProgramState.valueOf(exeComp.getString("State"));
			if(state == ProgramState.RUNNING) {

				// Read runtime environment.
				this.programCounter = exeComp.getInt("ProgramCounter");
				this.tickCount = exeComp.getInt("TickCount");
				this.actionDurationTicks = (exeComp.contains("ActionDurationTicks")
						? exeComp.getInt("ActionDurationTicks") : TICKS_PER_ACTION);
				CompoundNBT varMapComp = exeComp.getCompound("VarMap");
				for(String key : varMapComp.keySet()) {
					INBT nbtVal = varMapComp.get(key);
					Object val;
					if(nbtVal instanceof StringNBT) {
						val = ((StringNBT) nbtVal).getString();
					} else if(nbtVal instanceof IntNBT) {
						val = ((IntNBT) nbtVal).getInt();
					} else if(nbtVal instanceof ByteNBT) {
						val = ((ByteNBT) nbtVal).getByte() != 0;
					} else {
						LOGGER.warn(
								"Found unsupported type in variable map while loading from NBT: " + nbtVal.getClass());
						continue;
					}
					this.variableMap.put(key, val);
				}

				// Compile the program.
				CompileResult compileResult = null;
				try {
					compileResult = TurtleLangCompiler.compile(
							this.tile.getInventoryManager().getCodingInventory().getInventory());
				} catch (SyntaxException e) {

					// Reset when compilation fails.
					this.instructions = null;
					this.variableMap.clear();
				}
				if(compileResult != null) {

					// Compile the AST to executable instructions and reset the runtime environment.
					this.instructions = this.compileToInstructions(compileResult.getAST());

					// Reset the program counter and stop the turtle if the instruction index is invalid.
					if(this.programCounter < 0 || this.programCounter >= this.instructions.size()) {
						this.instructions = null;
						this.variableMap.clear();
					}
				}
			}
			this.setProgramState(state);
		}
	}

	/**
	 * Writes the execution state to NBT.
	 * @param nbt - The NBT compound to write to.
	 */
	public void writeToNBT(CompoundNBT nbt) {
		CompoundNBT exeComp = new CompoundNBT();
		exeComp.putString("State", this.state.name());
		if(this.state == ProgramState.RUNNING) {
			exeComp.putInt("ProgramCounter", this.programCounter);
			exeComp.putInt("TickCount", this.tickCount);
			if(this.actionDurationTicks != TICKS_PER_ACTION) {
				exeComp.putInt("ActionDurationTicks", this.actionDurationTicks);
			}
			CompoundNBT varMapComp = new CompoundNBT();
			for(Entry<String, Object> entry : this.variableMap.entrySet()) {
				Object val = entry.getValue();
				if(val instanceof String) {
					varMapComp.putString(entry.getKey(), (String) val);
				} else if(val instanceof Integer) {
					varMapComp.putInt(entry.getKey(), (Integer) val);
				} else if(val instanceof Boolean) {
					varMapComp.putBoolean(entry.getKey(), (Boolean) val);
				} else {
					LOGGER.warn("Found unsupported type in variable map while writing to NBT: " + val.getClass());
				}
			}
			exeComp.put("VarMap", varMapComp);
		}
		nbt.put("ExecutionState", exeComp);
	}

	public ProgramState getState() {
		return this.state;
	}

	/**
	 * Compiles the turtle code and schedules it to execute.
	 * Calling this while the program is already running will stop the current program and start the new program.
	 */
	public void start() {

		// Stop the current program if any.
		if(this.state == ProgramState.RUNNING) {
			this.setProgramState(ProgramState.STOPPED);
		}

		// Compile the program.
		CompileResult compileResult;
		try {
			compileResult = TurtleLangCompiler.compile(
					this.tile.getInventoryManager().getCodingInventory().getInventory());
		} catch (SyntaxException e) {

			// Reset when compilation fails.
			this.instructions = null;
			this.variableMap.clear();
			return;
		}

		// Compile the AST to executable instructions and reset the runtime environment.
		this.instructions = this.compileToInstructions(compileResult.getAST());
		this.variableMap.clear();
		this.programCounter = 0;
		this.tickCount = TICKS_PER_ACTION - TICKS_BEFORE_START;
		this.setProgramState(ProgramState.RUNNING);
	}

	/**
	 * Stops the current execution or resets from errored state to stopped state. Does nothing if the program was
	 * already stopped.
	 */
	public void stop() {
		this.setProgramState(ProgramState.STOPPED);
	}

	/**
	 * Compiles the given {@link CompiledProgram} to a single list of instructions, rather than a tree of terms.
	 * This allows for step-wise execution using a program counter.
	 * @param ast - The program to compile.
	 * @return A list of instructions.
	 */
	private List<Statement> compileToInstructions(Program ast) {
		return this.flattenStatementBlock(ast.statements, 0);
	}

	private List<Statement> flattenStatementBlock(Statements stmts, int numLoopVars) {
		List<Statement> statements = stmts.statements;
		for(int i = 0; i < statements.size(); i++) {
			Statement stmt = statements.get(i);
			if(stmt instanceof IfStatement) {
				IfStatement ifStmt = (IfStatement) stmt;

				// Get fields.
				Expression cond = ifStmt.condExp;
				Statements ifCode = ifStmt.ifCodeStmts;
				Statements elseCode = ifStmt.elseCodeStmts;

				// Keep a list of indices at which to put a "goto END" instruction.
				List<Integer> jumptoEndInstrIndices = new ArrayList<>();

				// Handle ifCode.
				List<Statement> ifCodeStmts = this.flattenStatementBlock(ifCode, numLoopVars);
				statements.set(i, new JumptoStatement(
						new UnaryExpression(-1, -1, UnaryOperator.NOT, cond), ifCodeStmts.size() + 2));
				for(Statement ifCodeStmt : ifCodeStmts) {
					statements.add(++i, ifCodeStmt);
				}
				statements.add(++i, null);
				jumptoEndInstrIndices.add(i);

				// Handle elseCode.
				if(elseCode != null) {
					List<Statement> elseCodeStmts = this.flattenStatementBlock(elseCode, numLoopVars);
					for(Statement elseCodeStmt : elseCodeStmts) {
						statements.add(++i, elseCodeStmt);
					}
				}

				// Insert "jumpto END" instructions.
				int endInd = i + 1;
				for(int ind : jumptoEndInstrIndices) {
					statements.set(ind, new JumptoStatement(new BooleanConstant(-1, true), endInd - ind));
				}

			} else if(stmt instanceof BreakStatement) {
				// Break statements are handled within loops that support them.
			} else if(stmt instanceof ForStatement) {
				ForStatement forStmt = (ForStatement) stmt;
				/*
				 * Incrementing: for(loopVar = startExp; loopVar <= endExp; loopVar++) { loopCode }
				 * Decrementing: for(loopVar = startExp; loopVar >= endExp; loopVar--) { loopCode }
				 * Implementation that supports incrementing and decrementing:
				 *     increment = startExp <= endExp;
				 *     loopVar = startExp;
				 *     loop: {
				 *         loopVar = internalLoopVar;
				 *         loopCode;
				 *         if(internalLoopVar == endExp) {
				 *             jumpto end;
				 *         }
				 *         if(increment) {
				 *             internalLoopVar = internalLoopVar + 1;
				 *         } else {
				 *             internalLoopVar = internalLoopVar - 1;
				 *         }
				 *         jumpto loop;
				 *     }
				 *     end:
				 */
				String loopVarName = forStmt.varName;
				Expression startExp = forStmt.startExp;
				Expression endExp = forStmt.endExp;
				Statements loopCode = forStmt.codeStmts;

				// "increment = startExp <= endExp;".
				String incrementVarName = "LOOPVAR~" + numLoopVars++;
				statements.set(i, new AssignStatement(-1, -1, incrementVarName,
						new BinaryExpression(-1, -1, BinaryOperator.LTE, startExp, endExp)));

				// "internalLoopVar = startExp;".
				String internalLoopVarName = "LOOPVAR~" + numLoopVars++;
				statements.add(++i, new AssignStatement(-1, -1, internalLoopVarName, startExp));

				// "loopVar = internalLoopVar;".
				statements.add(++i, new AssignStatement(-1, -1, loopVarName, new Variable(-1, internalLoopVarName)));

				// Flatten loopCode.
				List<Statement> loopStmts = this.flattenStatementBlock(loopCode, numLoopVars);

				// Handle breaks in loop code.
				int endJumpIndex = i + loopStmts.size() + 7;
				this.flattenBreakStatements(loopStmts, i + 1, endJumpIndex);

				// Insert loop code.
				for(Statement loopStmt : loopStmts) {
					statements.add(++i, loopStmt);
				}

				// "if(internalLoopVar == endExp) { jumpto end; }".
				int relEndJumpIndex = 6;
				Expression cond = new BinaryExpression(
						-1, -1, BinaryOperator.EQUALS, new Variable(-1, internalLoopVarName), endExp);
				statements.add(++i, new JumptoStatement(cond, relEndJumpIndex));

				// "if(increment) { internalLoopVar++; } else { internalLoopVar--; }"
				statements.add(++i, new JumptoStatement(new Variable(-1, incrementVarName), 3));
				statements.add(++i, new AssignStatement(-1, -1, internalLoopVarName, new BinaryExpression(
						-1, -1, BinaryOperator.MINUS,
						new Variable(-1, internalLoopVarName), new IntegerConstant(-1, 1))));
				statements.add(++i, new JumptoStatement(new BooleanConstant(-1, true), 2));
				statements.add(++i, new AssignStatement(-1, -1, internalLoopVarName, new BinaryExpression(
						-1, -1, BinaryOperator.PLUS,
						new Variable(-1, internalLoopVarName), new IntegerConstant(-1, 1))));

				// "jumpto loop;".
				statements.add(++i, new JumptoStatement(new BooleanConstant(-1, true), -loopStmts.size() - 6));

				// Remove loopVar and internalLoopVar (out of scope).
				statements.add(++i, new AssignStatement(-1, -1, loopVarName, null));
				statements.add(++i, new AssignStatement(-1, -1, internalLoopVarName, null));

			} else if(stmt instanceof RepeatStatement) {
				RepeatStatement repStmt = (RepeatStatement) stmt;
				Expression countExp = repStmt.repeatExp;
				Statements loopCode = repStmt.codeStmts;

				// "localLoopVar = countExp;".
				String localLoopVarName = "LOOPVAR~" + numLoopVars++;
				statements.set(i, new AssignStatement(-1, -1, localLoopVarName, countExp));

				// Flatten loopCode.
				List<Statement> loopStmts = this.flattenStatementBlock(loopCode, numLoopVars);

				// "if(localLoopVar <= 0) { jumpto END; }".
				int relEndJumpIndex = loopStmts.size() + 3;
				Expression cond = new BinaryExpression(
						-1, -1, BinaryOperator.LTE, new Variable(-1, localLoopVarName), new IntegerConstant(-1, 0));
				statements.add(++i, new JumptoStatement(cond, relEndJumpIndex));

				// Handle breaks in loop code.
				int endJumpIndex = i + loopStmts.size() + 3;
				this.flattenBreakStatements(loopStmts, i + 1, endJumpIndex);

				// Insert loop code.
				for(Statement loopStmt : loopStmts) {
					statements.add(++i, loopStmt);
				}

				// "localLoopVar = localLoopVar - 1;".
				statements.add(++i, new AssignStatement(-1, -1, localLoopVarName, new BinaryExpression(-1, -1,
								BinaryOperator.MINUS, new Variable(-1, localLoopVarName), new IntegerConstant(-1, 1))));

				// "jumpto cond_check;".
				statements.add(++i, new JumptoStatement(new BooleanConstant(-1, true),
						-2 - loopStmts.size()));

			} else if(stmt instanceof WhileStatement) {
				WhileStatement whileStmt = (WhileStatement) stmt;

				// Get fields.
				Expression cond = whileStmt.condExp;
				Statements loopCode = whileStmt.codeStmts;

				// Flatten loopCode.
				List<Statement> loopStmts = this.flattenStatementBlock(loopCode, numLoopVars);

				// "if(!cond) { jumpto END; }".
				int relEndJumpIndex = loopStmts.size() + 2;
				statements.set(i, new JumptoStatement(
						new UnaryExpression(-1, -1, UnaryOperator.NOT, cond), relEndJumpIndex));

				// Handle breaks in loop code.
				int endJumpIndex = i + loopStmts.size() + 2;
				this.flattenBreakStatements(loopStmts, i + 1, endJumpIndex);

				// "loopCode".
				for(Statement loopStmt : loopStmts) {
					statements.add(++i, loopStmt);
				}

				// "jumpto cond_check;".
				statements.add(++i, new JumptoStatement(new BooleanConstant(-1, true), -1 - loopStmts.size()));
			} else if(stmt instanceof CommentStatement) {
				
				// Remove comments, they don't have side effects.
				statements.remove(i--);
			} else if(stmt instanceof FunctionCallStatement) {
				FunctionCallStatement funcCallStmt = (FunctionCallStatement) stmt;

				// Flatten out function multi calls.
				if(funcCallStmt.callAmount == 0) {
					statements.remove(i--); // Remove functions with a zero call amount.
				} else if(funcCallStmt.callAmount > 1) {

					// Convert the current function call to a single call and insert it callAmount-1 extra times.
					int callAmount = funcCallStmt.callAmount;
					funcCallStmt.callAmount = 1;
					for(int j = 1; j < callAmount; j++) {
						statements.add(++i, funcCallStmt);
					}
				}
			}
		}
		return statements;
	}

	/**
	 * Replaces {@link BreakStatement}s by a jump to their goal. This should be called by loops that support breaking.
	 * @param stmts - The loop code statements.
	 * @param offset - The index at which the instructions in this list will be inserted into the final list of
	 * instructions.
	 * @param breakGoalIndex - The absolute index that should be jumped to (in the final list of instructions).
	 */
	private void flattenBreakStatements(List<Statement> stmts, int offset, int breakGoalIndex) {
		for(int i = 0; i < stmts.size(); i++) {
			Statement stmt = stmts.get(i);
			if(stmt instanceof BreakStatement) {
				stmts.set(i, new JumptoStatement(new BooleanConstant(-1, true), breakGoalIndex - offset - i));
			}
		}
	}

	@Override
	public void tick() {
		if(this.state == ProgramState.RUNNING && this.tickCount++ >= this.actionDurationTicks) {
			this.tickCount -= this.actionDurationTicks;
			this.step();
		}
	}

	/**
	 * Take steps in the program until some turtle action is executed.
	 * Stops the program if an error has occurred.
	 */
	private void step() {

		// Stop if no more instructions are available.
		// This triggers one tick late, which allows a turtle to finish its animation if it is doing one.
		if(this.instructions == null || this.programCounter >= this.instructions.size()) {
			this.setProgramState(ProgramState.STOPPED);
			return;
		}

		// Handle the next statement.
		int instrsExecuted = 0;
		try {
			while(this.evalStatement(this.instructions.get(this.programCounter))) {
				if(this.programCounter >= this.instructions.size()) {
					this.setProgramState(ProgramState.STOPPED);
					return;
				}
				if(++instrsExecuted > MAX_INSTRUCTIONS_PER_TICK) {
					throw new ExecutionException(new TranslationTextComponent(
							"compiler.computercraftsc.max_instructions_per_tick_exceeded_exception_message"));
				}
			}
		} catch (ExecutionException e) {
			IFormattableTextComponent textComponent = e.getTextComponent();
			this.setProgramState(ProgramState.ERRORED, (textComponent != null
					? textComponent : new StringTextComponent(e.getMessage())));
		} catch (Throwable t) {
			t.printStackTrace();
			// TODO - Reconsider after testing. This causes internal errors to be displayed to players.
			this.setProgramState(ProgramState.ERRORED,
					new StringTextComponent("Internal error: " + t.getClass().getSimpleName() + ": " + t.getMessage()));
		}
	}

	/**
	 * Evaluates the given statement.
	 * @param stmt - The statement.
	 * @return {@code true} if the next statement should evaluate instantly, {@code false} if the next statement should
	 * run after a break. This should return {@code false} for statements that cause turtles to perform some
	 * animated action.
	 * @throws RuntimeTypeException If a type error is encountered.
	 * @throws ExecutionException If an error occurs while evaluating the statement.
	 */
	private boolean evalStatement(Statement stmt) throws RuntimeTypeException, ExecutionException {
		
		// Set default action duration.
		this.actionDurationTicks = TICKS_PER_ACTION;
		
		// Evaluate statement.
		if(stmt instanceof FunctionCallStatement) {
			FunctionCallStatement fcStmt = (FunctionCallStatement) stmt;
			Expression[] args = fcStmt.args;
			this.programCounter++;
			
			ITurtleCommand command;
			switch(fcStmt.function) {
				case MOVE_FORWARD: {
					command = new TurtleMoveCommand(MoveDirection.FORWARD);
					break;
				}
				case MOVE_BACK: {
					command = new TurtleMoveCommand(MoveDirection.BACK);
					break;
				}
				case TURN_LEFT: {
					command = new TurtleTurnCommand(TurnDirection.LEFT);
					break;
				}
				case TURN_RIGHT: {
					command = new TurtleTurnCommand(TurnDirection.RIGHT);
					break;
				}
				case MOVE_UP: {
					command = new TurtleMoveCommand(MoveDirection.UP);
					break;
				}
				case MOVE_DOWN: {
					command = new TurtleMoveCommand(MoveDirection.DOWN);
					break;
				}
				case PLACE: {
					command = new TurtlePlaceCommand(InteractDirection.FORWARD, null);
					break;
				}
				case PLACE_UP: {
					command = new TurtlePlaceCommand(InteractDirection.UP, null);
					break;
				}
				case PLACE_DOWN: {
					command = new TurtlePlaceCommand(InteractDirection.DOWN, null);
					break;
				}
				case DIG: {
					command = TurtleToolCommand.dig(InteractDirection.FORWARD, TurtleSide.RIGHT);
					break;
				}
				case DIG_UP: {
					command = TurtleToolCommand.dig(InteractDirection.UP, TurtleSide.RIGHT);
					break;
				}
				case DIG_DOWN: {
					command = TurtleToolCommand.dig(InteractDirection.DOWN, TurtleSide.RIGHT);
					break;
				}
				case DROP: {
					int numItems = 64;
					command = new TurtleDropCommand(InteractDirection.FORWARD, numItems);
					break;
				}
				case DROP_UP: {
					int numItems = 64;
					command = new TurtleDropCommand(InteractDirection.UP, numItems);
					break;
				}
				case DROP_DOWN: {
					int numItems = 64;
					command = new TurtleDropCommand(InteractDirection.DOWN, numItems);
					break;
				}
				case SUCK: {
					int numItems = 64;
					command = new TurtleSuckCommand(InteractDirection.FORWARD, numItems);
					break;
				}
				case SUCK_UP: {
					int numItems = 64;
					command = new TurtleSuckCommand(InteractDirection.UP, numItems);
					break;
				}
				case SUCK_DOWN: {
					int numItems = 64;
					command = new TurtleSuckCommand(InteractDirection.DOWN, numItems);
					break;
				}
				case ATTACK: {
					command = TurtleToolCommand.attack(InteractDirection.FORWARD, TurtleSide.RIGHT);
					break;
				}
				case ATTACK_UP: {
					command = TurtleToolCommand.attack(InteractDirection.UP, TurtleSide.RIGHT);
					break;
				}
				case ATTACK_DOWN: {
					command = TurtleToolCommand.attack(InteractDirection.DOWN, TurtleSide.RIGHT);
					break;
				}
				case SET_REDSTONE: {
					boolean powered = requireBool(this.evalExpression(args[0]));
					int redstonePower = (powered ? 15 : 0);
					this.tile.setRedstoneOutput(InteractDirection.FORWARD, redstonePower);
					return false;
				}
				case SET_REDSTONE_UP: {
					boolean powered = requireBool(this.evalExpression(args[0]));
					int redstonePower = (powered ? 15 : 0);
					this.tile.setRedstoneOutput(InteractDirection.UP, redstonePower);
					return false;
				}
				case SET_REDSTONE_DOWN: {
					boolean powered = requireBool(this.evalExpression(args[0]));
					int redstonePower = (powered ? 15 : 0);
					this.tile.setRedstoneOutput(InteractDirection.DOWN, redstonePower);
					return false;
				}
				case SAY: {
					Object textObj = this.evalExpression(args[0]);
					String text = (textObj == null ? null : textObj.toString());
					this.tile.getWorld().getServer().getPlayerList().func_232641_a_(
							new TranslationTextComponent("chat.type.announcement", 
									new Object[] { new StringTextComponent("Turtle"), new StringTextComponent(text) }),
							ChatType.SYSTEM, Util.DUMMY_UUID);
					return false;
				}
				case SELECT: {
					int slot = requireInt(this.evalExpression(args[0]));
					if(slot < 0 || slot > 15) {
						throw new ExecutionException(new TranslationTextComponent("compiler.computercraftsc"
								+ ".slot_number_out_of_bounds_exception_message__min_index__max_index", 0, 15));
					}
					this.tile.getAccess().setSelectedSlot(slot);
					return true;
				}
				case SLEEP: {
					this.actionDurationTicks = requireInt(this.evalExpression(args[0]));
					return false;
				}
				default: {
					throw new RuntimeTypeException("Unsupported statement function type: " + fcStmt.function);
				}
			}
			TurtleCommandResult result = command.execute(this.tile.getAccess());
			if(!result.isSuccess()) {
				
				// Translate known result error messages from CC-Tweaked.
				switch(result.getErrorMessage()) {
					case "Movement obstructed": {
						throw new ExecutionException(new TranslationTextComponent(
								"compiler.computercraftsc.command_exception_movement_obstructed"));
					}
					case "No items to drop": {
						throw new ExecutionException(new TranslationTextComponent(
								"compiler.computercraftsc.command_exception_no_items_to_drop"));
					}
					case "No space for items": {
						throw new ExecutionException(new TranslationTextComponent(
								"compiler.computercraftsc.command_exception_no_space_for_items"));
					}
					case "No block to inspect": {
						throw new ExecutionException(new TranslationTextComponent(
								"compiler.computercraftsc.command_exception_no_block_to_inspect"));
					}
					case "Movement failed": {
						throw new ExecutionException(new TranslationTextComponent(
								"compiler.computercraftsc.command_exception_movement_failed"));
					}
					case "No items to place": {
						throw new ExecutionException(new TranslationTextComponent(
								"compiler.computercraftsc.command_exception_no_items_to_place"));
					}
					case "No items to take": {
						throw new ExecutionException(new TranslationTextComponent(
								"compiler.computercraftsc.command_exception_no_items_to_take"));
					}
					case "Cannot place item here": {
						throw new ExecutionException(new TranslationTextComponent(
								"compiler.computercraftsc.command_exception_cannot_place_item_here"));
					}
					case "Unbreakable block detected": {
						throw new ExecutionException(new TranslationTextComponent(
								"compiler.computercraftsc.command_exception_unbreakable_block_detected"));
					}
					case "Cannot break block with this tool": {
						throw new ExecutionException(new TranslationTextComponent(
								"compiler.computercraftsc.command_exception_cannot_break_block_with_this_tool"));
					}
					case "Nothing to attack here": {
						throw new ExecutionException(new TranslationTextComponent(
								"compiler.computercraftsc.command_exception_nothing_to_attack_here"));
					}
					case "Cannot break protected block": {
						throw new ExecutionException(new TranslationTextComponent(
								"compiler.computercraftsc.command_exception_cannot_break_protected_block"));
					}
					default: {
						throw new ExecutionException(new StringTextComponent(result.getErrorMessage()));
					}
				}
			}
			return false;
		} else if(stmt instanceof JumptoStatement) {
			JumptoStatement jumptoStmt = (JumptoStatement) stmt;
			Object retVal = this.evalExpression(jumptoStmt.condExp);
			if(retVal instanceof Boolean) {
				boolean cond = ((Boolean) retVal).booleanValue();
				this.programCounter += (cond ? jumptoStmt.relJump : 1);
			} else {
				throw new RuntimeTypeException("Invalid jumpto condition: "
						+ (retVal == null ? "~null" : retVal.getClass().getSimpleName()));
			}
			return true;
		} else if(stmt instanceof AssignStatement) {
			AssignStatement assignStmt = (AssignStatement) stmt;
			if(assignStmt.valExp == null) {
				this.variableMap.remove(assignStmt.variableName);
			} else {
				Object val = this.evalExpression(assignStmt.valExp);
				if(val == null) {
					throw new RuntimeTypeException(new TranslationTextComponent(
							"compiler.computercraftsc.cannot_assign_void_to_variables_exception_message"));
				}
				this.variableMap.put(assignStmt.variableName, val);
			}
			this.programCounter++;
			return true;
		} else if(stmt instanceof BreakStatement) {
			// This is impossible if the code validation and compiler works correctly.
			throw new Error("Unconverted break statement found during turtle execution.");
		}
		throw new ExecutionException("Unsupported statement in eval: " + stmt.getClass().getSimpleName());
	}
	
	/**
	 * Evaluates the given expression.
	 * @param expr - The expression.
	 * @return The result of the expression. This can be one of the following: String, Integer, Boolean, null.
	 * If evaluation of an expression results in {@code null}, then it should be considered a statement.
	 * @throws RuntimeTypeException If a type error is encountered.
	 * @throws ExecutionException If an error occurs while evaluating the expression.
	 */
	private Object evalExpression(Expression expr) throws RuntimeTypeException, ExecutionException {
		if(expr instanceof FunctionCallExpression) {
			FunctionCallExpression fcExpr = (FunctionCallExpression) expr;
			Expression[] args = fcExpr.args;

			ITurtleCommand command;
			switch(fcExpr.function) {
				case RANDOM_NUMBER: {
					int bound = requireInt(this.evalExpression(args[0]));
					return new Random().nextInt(bound);
				}
				case RANDOM_BOOL: {
					return new Random().nextBoolean();
				}
				case GET_ITEM_COUNT: {
					int selectedSlot = this.tile.getAccess().getSelectedSlot();
					ItemStack stack = this.tile.getInventoryManager().getTurtleInventory().getStackInSlot(selectedSlot);
					return (stack == null ? 0 : stack.getCount());
				}
				case DETECT: {
					command = new TurtleDetectCommand(InteractDirection.FORWARD);
					return command.execute(this.tile.getAccess()).isSuccess();
				}
				case DETECT_UP: {
					command = new TurtleDetectCommand(InteractDirection.UP);
					return command.execute(this.tile.getAccess()).isSuccess();
				}
				case DETECT_DOWN: {
					command = new TurtleDetectCommand(InteractDirection.DOWN);
					return command.execute(this.tile.getAccess()).isSuccess();
				}
				case COMPARE: {
					command = new TurtleCompareCommand(InteractDirection.FORWARD);
					return command.execute(this.tile.getAccess()).isSuccess();
				}
				case COMPARE_UP: {
					command = new TurtleCompareCommand(InteractDirection.UP);
					return command.execute(this.tile.getAccess()).isSuccess();
				}
				case COMPARE_DOWN: {
					command = new TurtleCompareCommand(InteractDirection.DOWN);
					return command.execute(this.tile.getAccess()).isSuccess();
				}
				case INSPECT: {
					command = new TurtleInspectCommand(InteractDirection.FORWARD);
					TurtleCommandResult result = command.execute(this.tile.getAccess());
					if(!result.isSuccess()) {
						return "minecraft:air"; // This command can only fail when inspecting air, so return air here.
					}
					return (String) ((Map<?, ?>) result.getResults()[0]).get("name");
				}
				case INSPECT_UP: {
					command = new TurtleInspectCommand(InteractDirection.UP);
					TurtleCommandResult result = command.execute(this.tile.getAccess());
					if(!result.isSuccess()) {
						return "minecraft:air"; // This command can only fail when inspecting air, so return air here.
					}
					return (String) ((Map<?, ?>) result.getResults()[0]).get("name");
				}
				case INSPECT_DOWN: {
					command = new TurtleInspectCommand(InteractDirection.DOWN);
					TurtleCommandResult result = command.execute(this.tile.getAccess());
					if(!result.isSuccess()) {
						return "minecraft:air"; // This command can only fail when inspecting air, so return air here.
					}
					return (String) ((Map<?, ?>) result.getResults()[0]).get("name");
				}
				case INSPECT_SLOT: {
					int selectedSlot = this.tile.getAccess().getSelectedSlot();
					ItemStack stack = this.tile.getInventoryManager().getTurtleInventory().getStackInSlot(selectedSlot);
					Item item = (stack != null && stack.getCount() > 0 ? stack.getItem() : Items.AIR);
					return item.getRegistryName().getNamespace() + ":" + item.getRegistryName().getPath();
				}
				case QUERY_REDSTONE: {
					return this.tile.getRedstoneInput(InteractDirection.FORWARD) > 0;
				}
				case QUERY_REDSTONE_UP: {
					return this.tile.getRedstoneInput(InteractDirection.UP) > 0;
				}
				case QUERY_REDSTONE_DOWN: {
					return this.tile.getRedstoneInput(InteractDirection.DOWN) > 0;
				}
				default: {
					throw new RuntimeTypeException("Unsupported expression function type: " + fcExpr.function);
				}
			}
		} else if(expr instanceof BooleanConstant) {
			return ((BooleanConstant) expr).value;
		} else if(expr instanceof IntegerConstant) {
			return ((IntegerConstant) expr).value;
		} else if(expr instanceof StringConstant) {
			return ((StringConstant) expr).value;
		} else if(expr instanceof Variable) {
			Variable var = (Variable) expr;
			Object val = this.variableMap.get(var.name);
			if(val == null) {
				throw new RuntimeTypeException(new TranslationTextComponent(
						"compiler.computercraftsc.undefined_variable_exception_message__variable_name"));
			}
			return val;
		} else if(expr instanceof BinaryExpression) {
			BinaryExpression binExpr = (BinaryExpression) expr;
			
			// Evaluate the left and right expressions.
			Object leftVal = this.evalExpression(binExpr.leftExp);
			Object rightVal = this.evalExpression(binExpr.rightExp);
			
			// Get the operator.
			BinaryOperator operator = binExpr.operator;
			
			// Throw exception for null values. This shouldn't be possible.
			if(leftVal == null) {
				throw new RuntimeTypeException(
						"Found a null value on the left side of " + operator.getRepresentation() + ".");
			}
			if(rightVal == null) {
				throw new RuntimeTypeException(
						"Found a null value on the right side of " + operator.getRepresentation() + ".");
			}
			
			// Evaluate the proper operator.
			switch(operator) {
				case EQUALS: return leftVal.equals(rightVal);
				case NOT_EQUALS: return !leftVal.equals(rightVal);
				case LT: return requireInt(leftVal) < requireInt(rightVal);
				case LTE: return requireInt(leftVal) <= requireInt(rightVal);
				case GT: return requireInt(leftVal) > requireInt(rightVal);
				case GTE: return requireInt(leftVal) >= requireInt(rightVal);
				case PLUS: return requireInt(leftVal) + requireInt(rightVal);
				case MINUS: return requireInt(leftVal) - requireInt(rightVal);
				case TIMES: return requireInt(leftVal) * requireInt(rightVal);
				case DIV: {
					int right = requireInt(rightVal);
					if(right == 0) {
						throw new RuntimeTypeException(new TranslationTextComponent(
								"compiler.computercraftsc.division_by_zero_exception_message"));
					}
					return requireInt(leftVal) / right;
				}
				case AND: return requireBool(leftVal) && requireBool(rightVal);
				case OR: return requireBool(leftVal) || requireBool(rightVal);
				default: {
					throw new RuntimeTypeException(
							"Unsupported operator in binary expression: " + operator.getRepresentation());
				}
			}
		} else if(expr instanceof UnaryExpression) {
			UnaryExpression unExpr = (UnaryExpression) expr;
			
			// Evaluate the expressions.
			Object val = this.evalExpression(unExpr.exp);
			
			// Get the operator.
			UnaryOperator operator = unExpr.operator;
			
			// Throw exception for null values. This shouldn't be possible.
			if(val == null) {
				throw new RuntimeTypeException("Found a null value in operator " + operator.getRepresentation() + ".");
			}
			
			// Evaluate the proper operator.
			switch(operator) {
				case NOT: return !requireBool(val);
				case POSITIVE: return requireInt(val);
				case NEGATIVE: return -requireInt(val);
				default: {
					throw new RuntimeTypeException(
							"Unsupported operator in unary expression: " + operator.getRepresentation());
				}
			}
		}
		throw new RuntimeTypeException("Unsupported expression in eval: " + expr.getClass().getSimpleName());
	}

	private void setProgramState(ProgramState state, IFormattableTextComponent errorMessage) {
		if(this.state != state) {
			this.tile.onProgramStateChanged(this.state, state, errorMessage);
			ProgramStateChangedEvent event = new ProgramStateChangedEvent(this.tile, this.state, state, errorMessage);
			EventManager.getInstance().fireProgStateChangedEvent(event);
			if(!event.isCancelled()) {
				this.state = state;
			}
		}
	}

	private void setProgramState(ProgramState state) {
		this.setProgramState(state, null);
	}

	private static int requireInt(Object obj) throws RuntimeTypeException {
		if(obj instanceof Integer) {
			return (Integer) obj;
		}
		throw new RuntimeTypeException(new TranslationTextComponent(
				"compiler.computercraftsc.type_exception_message__expected_type__actual_type",
				"Integer", (obj == null ? "null" : obj.getClass().getSimpleName())));
	}

	private static boolean requireBool(Object obj) throws RuntimeTypeException {
		if(obj instanceof Boolean) {
			return (Boolean) obj;
		}
		throw new RuntimeTypeException(new TranslationTextComponent(
				"compiler.computercraftsc.type_exception_message__expected_type__actual_type",
				"Boolean", (obj == null ? "null" : obj.getClass().getSimpleName())));
	}

	@SuppressWarnings("unused") // Reasoning: This method makes the 'require' methods complete for all types, and could have a use in the future.
	private static String requireString(Object obj) throws RuntimeTypeException {
		if(obj instanceof String) {
			return (String) obj;
		}
		throw new RuntimeTypeException(new TranslationTextComponent(
				"compiler.computercraftsc.type_exception_message__expected_type__actual_type",
				"String", (obj == null ? "null" : obj.getClass().getSimpleName())));
	}
}
