package computercraftsc.shared.turtle.core.code.ast;

import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

import computercraftsc.shared.RegistrySC.ModItems;
import computercraftsc.shared.items.ItemBlockName;
import computercraftsc.shared.items.ItemItemName;
import computercraftsc.shared.items.ItemNumber;
import computercraftsc.shared.items.ItemString;
import computercraftsc.shared.items.ItemVariable;
import computercraftsc.shared.turtle.core.code.compiler.exception.ExpectedASTTermSyntaxException;
import computercraftsc.shared.turtle.core.code.compiler.exception.SyntaxException;
import net.minecraft.item.ItemStack;

/**
 * Represents an expression, being a term that can be used as a value.
 * @author P.J.S. Kools
 */
public abstract class Expression extends ASTTerm {
	
	public Expression(int codeStartIndex, int codeEndIndex) {
		super(codeStartIndex, codeEndIndex);
	}
	
	public static Expression parse(List<ItemStack> code, AtomicInteger codeIndex) throws SyntaxException {
		
		// Parse simple expression (without binary expressions).
		Expression exp = parseSimpleExp(code, codeIndex);
		
		// Check whether the expression is the first term of a binary expression.
		ItemStack stack = getFirstNonNullItem(code, codeIndex);
		if(stack == null) {
			
			// The expression is not part of a binary expression.
			return exp;
		}
		
		BinaryOperator operator;
		if(stack.getItem() == ModItems.OR.get()) {
			operator = BinaryOperator.OR;
		} else if(stack.getItem() == ModItems.AND.get()) {
			operator = BinaryOperator.AND;
		} else if(stack.getItem() == ModItems.EQUAL_TO.get()) {
			operator = BinaryOperator.EQUALS;
		} else if(stack.getItem() == ModItems.NOT_EQUAL_TO.get()) {
			operator = BinaryOperator.NOT_EQUALS;
		} else if(stack.getItem() == ModItems.LESS_THAN.get()) {
			operator = BinaryOperator.LT;
		} else if(stack.getItem() == ModItems.LESS_THAN_OR_EQUAL_TO.get()) {
			operator = BinaryOperator.LTE;
		} else if(stack.getItem() == ModItems.GREATER_THAN.get()) {
			operator = BinaryOperator.GT;
		} else if(stack.getItem() == ModItems.GREATER_THAN_OR_EQUAL_TO.get()) {
			operator = BinaryOperator.GTE;
		} else if(stack.getItem() == ModItems.PLUS.get()) {
			operator = BinaryOperator.PLUS;
		} else if(stack.getItem() == ModItems.MINUS.get()) {
			operator = BinaryOperator.MINUS;
		} else if(stack.getItem() == ModItems.TIMES.get()) {
			operator = BinaryOperator.TIMES;
		} else if(stack.getItem() == ModItems.DIVIDED_BY.get()) {
			operator = BinaryOperator.DIV;
		} else {
			
			// The expression is not part of a binary expression.
			return exp;
		}
		codeIndex.getAndIncrement(); // Skip over operator index.
		
		// Parse the second expression of this binary expression.
		Expression rightExp = Expression.parse(code, codeIndex);
		
		// Account for operator precedence when the second expression is a binary expression as well.
		Expression otherExp = rightExp;
		BinaryExpression prevOtherExp = null;
		while(otherExp instanceof BinaryExpression
				&& operator.hasPrecedenceOver(((BinaryExpression) otherExp).operator)) {
			prevOtherExp = (BinaryExpression) otherExp;
			otherExp = prevOtherExp.leftExp;
		}
		if(prevOtherExp == null) {
			
			// This binary operator is either not nested or not a higher priority than the next operator.
			return new BinaryExpression(exp.getCodeStartIndex(), rightExp.getCodeEndIndex(), operator, exp, rightExp);
		} else {
			
			// This binary operator is nested and has a higher priority, so rewrite it in the proper position.
			prevOtherExp.leftExp = new BinaryExpression(exp.getCodeStartIndex(),
					prevOtherExp.leftExp.getCodeEndIndex(), operator, exp, prevOtherExp.leftExp);
			return rightExp;
		}
	}
	
	private static Expression parseSimpleExp(List<ItemStack> code, AtomicInteger codeIndex) throws SyntaxException {
		ItemStack stack = getFirstNonNullItem(code, codeIndex);
		if(stack == null) {
			throw new ExpectedASTTermSyntaxException(Expression.class, codeIndex.get());
		}
		
		// Handle boolean constants.
		if(stack.getItem() == ModItems.TRUE.get()) {
			return new BooleanConstant(codeIndex.getAndIncrement(), true);
		}
		if(stack.getItem() == ModItems.FALSE.get()) {
			return new BooleanConstant(codeIndex.getAndIncrement(), false);
		}
		
		// Handle integer constants.
		if(stack.getItem() == ModItems.NUMBER.get()) {
			ItemNumber numberItem = (ItemNumber) stack.getItem();
			return new IntegerConstant(codeIndex.getAndIncrement(), numberItem.getNumber(stack));
		}
		
		// Handle string constants.
		if(stack.getItem() == ModItems.STRING.get()) {
			ItemString stringItem = (ItemString) stack.getItem();
			return new StringConstant(codeIndex.getAndIncrement(), stringItem.getString(stack));
		}
		
		// Handle variables.
		if(stack.getItem() == ModItems.VARIABLE.get()) {
			ItemVariable variableItem = (ItemVariable) stack.getItem();
			return new Variable(codeIndex.getAndIncrement(), variableItem.getName(stack));
		}
		
		// Handle unary expressions.
		if(stack.getItem() == ModItems.NOT.get()) {
			int unExpIndex = codeIndex.getAndIncrement();
			Expression exp = Expression.parseSimpleExp(code, codeIndex);
			return new UnaryExpression(unExpIndex, exp.getCodeEndIndex(), UnaryOperator.NOT, exp);
		}
		if(stack.getItem() == ModItems.PLUS.get()) {
			int unExpIndex = codeIndex.getAndIncrement();
			Expression exp = Expression.parseSimpleExp(code, codeIndex);
			return new UnaryExpression(unExpIndex, exp.getCodeEndIndex(), UnaryOperator.POSITIVE, exp);
		}
		if(stack.getItem() == ModItems.MINUS.get()) {
			int unExpIndex = codeIndex.getAndIncrement();
			Expression exp = Expression.parseSimpleExp(code, codeIndex);
			return new UnaryExpression(unExpIndex, exp.getCodeEndIndex(), UnaryOperator.NEGATIVE, exp);
		}
		
		// Handle function expressions.
		if(stack.getItem() == ModItems.RANDOM_NUMBER.get()) {
			int randNumIndex = codeIndex.getAndIncrement();
			Expression exp = Expression.parseSimpleExp(code, codeIndex);
			return new FunctionCallExpression(randNumIndex, exp.getCodeEndIndex(), ExprFunctionType.RANDOM_NUMBER, exp);
		}
		if(stack.getItem() == ModItems.RANDOM_BOOLEAN.get()) {
			return new FunctionCallExpression(codeIndex.getAndIncrement(), ExprFunctionType.RANDOM_BOOL);
		}
		if(stack.getItem() == ModItems.GET_ITEM_COUNT.get()) {
			return new FunctionCallExpression(codeIndex.getAndIncrement(), ExprFunctionType.GET_ITEM_COUNT);
		}
		if(stack.getItem() == ModItems.DETECT.get()) {
			return new FunctionCallExpression(codeIndex.getAndIncrement(), ExprFunctionType.DETECT);
		}
		if(stack.getItem() == ModItems.DETECT_UP.get()) {
			return new FunctionCallExpression(codeIndex.getAndIncrement(), ExprFunctionType.DETECT_UP);
		}
		if(stack.getItem() == ModItems.DETECT_DOWN.get()) {
			return new FunctionCallExpression(codeIndex.getAndIncrement(), ExprFunctionType.DETECT_DOWN);
		}
		if(stack.getItem() == ModItems.COMPARE.get()) {
			return new FunctionCallExpression(codeIndex.getAndIncrement(), ExprFunctionType.COMPARE);
		}
		if(stack.getItem() == ModItems.COMPARE_UP.get()) {
			return new FunctionCallExpression(codeIndex.getAndIncrement(), ExprFunctionType.COMPARE_UP);
		}
		if(stack.getItem() == ModItems.COMPARE_DOWN.get()) {
			return new FunctionCallExpression(codeIndex.getAndIncrement(), ExprFunctionType.COMPARE_DOWN);
		}
		if(stack.getItem() == ModItems.INSPECT.get()) {
			return new FunctionCallExpression(codeIndex.getAndIncrement(), ExprFunctionType.INSPECT);
		}
		if(stack.getItem() == ModItems.INSPECT_UP.get()) {
			return new FunctionCallExpression(codeIndex.getAndIncrement(), ExprFunctionType.INSPECT_UP);
		}
		if(stack.getItem() == ModItems.INSPECT_DOWN.get()) {
			return new FunctionCallExpression(codeIndex.getAndIncrement(), ExprFunctionType.INSPECT_DOWN);
		}
		if(stack.getItem() == ModItems.INSPECT_SLOT.get()) {
			return new FunctionCallExpression(codeIndex.getAndIncrement(), ExprFunctionType.INSPECT_SLOT);
		}
		if(stack.getItem() == ModItems.BLOCK.get()) {
			return new StringConstant(codeIndex.getAndIncrement(), ItemBlockName.getBlockName(stack));
		}
		if(stack.getItem() == ModItems.ITEM.get()) {
			return new StringConstant(codeIndex.getAndIncrement(), ItemItemName.getItemName(stack));
		}
		if(stack.getItem() == ModItems.QUERY_REDSTONE.get()) {
			return new FunctionCallExpression(codeIndex.getAndIncrement(), ExprFunctionType.QUERY_REDSTONE);
		}
		if(stack.getItem() == ModItems.QUERY_REDSTONE_UP.get()) {
			return new FunctionCallExpression(codeIndex.getAndIncrement(), ExprFunctionType.QUERY_REDSTONE_UP);
		}
		if(stack.getItem() == ModItems.QUERY_REDSTONE_DOWN.get()) {
			return new FunctionCallExpression(codeIndex.getAndIncrement(), ExprFunctionType.QUERY_REDSTONE_DOWN);
		}
		
		// No expression found.
		throw new ExpectedASTTermSyntaxException(Expression.class, codeIndex.get());
	}
}
