package computercraftsc.shared.turtle.core.code.ast;

import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

import computercraftsc.shared.RegistrySC.ModItems;
import computercraftsc.shared.turtle.core.code.compiler.exception.CompileException;
import computercraftsc.shared.turtle.core.code.compiler.exception.ExpectedItemSyntaxException;
import computercraftsc.shared.turtle.core.code.compiler.exception.SyntaxException;
import computercraftsc.shared.turtle.core.code.type.ClassType;
import net.minecraft.item.ItemStack;

public class ForStatement extends Statement {
	
	public String varName;
	public Expression startExp;
	public Expression endExp;
	public Statements codeStmts;
	
	public ForStatement(int codeStartIndex, int codeEndIndex,
			String varName, Expression startExp, Expression endExp, Statements codeStmts) {
		super(codeStartIndex, codeEndIndex);
		this.varName = varName;
		this.startExp = startExp;
		this.endExp = endExp;
		this.codeStmts = codeStmts;
	}
	
	public static ForStatement parse(List<ItemStack> code, AtomicInteger codeIndex) throws SyntaxException {
		
		// Handle for term.
		ItemStack forStack = getFirstNonNullItem(code, codeIndex);
		if(forStack == null || forStack.getItem() != ModItems.FOR.get()) {
			throw new ExpectedItemSyntaxException(ModItems.FOR.get(), codeIndex.get());
		}
		int forIndex = codeIndex.get();
		
		// Handle variable term.
		codeIndex.incrementAndGet();
		ItemStack variableStack = getFirstNonNullItem(code, codeIndex);
		if(variableStack == null || variableStack.getItem() != ModItems.VARIABLE.get()) {
			throw new ExpectedItemSyntaxException(ModItems.VARIABLE.get(), codeIndex.get());
		}
		String variableName = (variableStack.hasDisplayName() ? variableStack.getDisplayName().getString() : null);
		
		// Handle "=" term.
		codeIndex.incrementAndGet();
		ItemStack equalsStack = getFirstNonNullItem(code, codeIndex);
		if(equalsStack == null || equalsStack.getItem() != ModItems.EQUAL_TO.get()) {
			throw new ExpectedItemSyntaxException(ModItems.EQUAL_TO.get(), codeIndex.get());
		}
		
		// Handle startExp term.
		codeIndex.incrementAndGet();
		Expression startExp = Expression.parse(code, codeIndex);
		
		// Handle to term.
		ItemStack toStack = getFirstNonNullItem(code, codeIndex);
		if(toStack == null || toStack.getItem() != ModItems.TO.get()) {
			throw new ExpectedItemSyntaxException(ModItems.TO.get(), codeIndex.get());
		}
		
		// Handle endExp term.
		codeIndex.incrementAndGet();
		Expression endExp = Expression.parse(code, codeIndex);
		
		// Handle do term.
		ItemStack doStack = getFirstNonNullItem(code, codeIndex);
		if(doStack == null || doStack.getItem() != ModItems.DO.get()) {
			throw new ExpectedItemSyntaxException(ModItems.DO.get(), codeIndex.get());
		}
		
		// Handle statements term.
		codeIndex.incrementAndGet();
		Statements codeStmts = Statements.parse(code, codeIndex, ModItems.END.get());
		
		// Handle end term.
		ItemStack endStack = getFirstNonNullItem(code, codeIndex);
		if(endStack == null || endStack.getItem() != ModItems.END.get()) {
			throw new ExpectedItemSyntaxException(ModItems.END.get(), codeIndex.get());
		}
		int endIndex = codeIndex.get();
		codeIndex.incrementAndGet();
		
		// Return the result.
		return new ForStatement(forIndex, endIndex, variableName, startExp, endExp, codeStmts);
	}
	
	@Override
	public String toCodeString() {
		return "for(" + this.varName + " = " + this.startExp.toCodeString() + " to " + this.endExp.toCodeString()
				+ ") {\n\t" + this.codeStmts.toCodeString().replaceAll("\n", "\n\t") + "\n}";
	}
	
	@Override
	public ClassType typecheck(List<CompileException> compileExceptions) {
		this.requireType(this.startExp, ClassType.INT, compileExceptions);
		this.requireType(this.endExp, ClassType.INT, compileExceptions);
		this.codeStmts.typecheck(compileExceptions);
		return ClassType.VOID;
	}
}
