package computercraftsc.shared.turtle.core.code.ast;

import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

import computercraftsc.shared.RegistrySC.ModItems;
import computercraftsc.shared.turtle.core.code.compiler.exception.CompileException;
import computercraftsc.shared.turtle.core.code.compiler.exception.ExpectedItemSyntaxException;
import computercraftsc.shared.turtle.core.code.compiler.exception.SyntaxException;
import computercraftsc.shared.turtle.core.code.type.ClassType;
import net.minecraft.item.ItemStack;

public class IfStatement extends Statement {
	
	public Expression condExp;
	public Statements ifCodeStmts;
	public Statements elseCodeStmts;
	
	public IfStatement(int codeStartIndex, int codeEndIndex,
			Expression condExp, Statements ifCodeStmts, Statements elseCodeStmts) {
		super(codeStartIndex, codeEndIndex);
		this.condExp = condExp;
		this.ifCodeStmts = ifCodeStmts;
		this.elseCodeStmts = elseCodeStmts;
	}
	
	public static IfStatement parse(List<ItemStack> code, AtomicInteger codeIndex) throws SyntaxException {
		
		// Handle if term.
		ItemStack ifStack = getFirstNonNullItem(code, codeIndex);
		if(ifStack == null || ifStack.getItem() != ModItems.IF.get()) {
			throw new ExpectedItemSyntaxException(ModItems.IF.get(), codeIndex.get());
		}
		int ifIndex = codeIndex.get();
		
		// Handle expression term.
		codeIndex.incrementAndGet();
		Expression condExp = Expression.parse(code, codeIndex);
		
		// Handle then term.
		ItemStack thenStack = getFirstNonNullItem(code, codeIndex);
		if(thenStack == null || thenStack.getItem() != ModItems.THEN.get()) {
			throw new ExpectedItemSyntaxException(ModItems.THEN.get(), codeIndex.get());
		}
		
		// Handle statements term.
		codeIndex.incrementAndGet();
		Statements ifCodeStmts = Statements.parse(code, codeIndex,
				ModItems.END.get(), ModItems.ELSE_IF.get(), ModItems.ELSE.get());
		
		// Handle optional else terms.
		Statements elseCodeStmts = parseOptionalElse(code, codeIndex);
		
		// Handle end term.
		ItemStack endStack = getFirstNonNullItem(code, codeIndex);
		if(endStack == null || endStack.getItem() != ModItems.END.get()) {
			throw new ExpectedItemSyntaxException(ModItems.END.get(), codeIndex.get());
		}
		int endIndex = codeIndex.get();
		codeIndex.incrementAndGet();
		
		// Return the result.
		return new IfStatement(ifIndex, endIndex, condExp, ifCodeStmts, elseCodeStmts);
	}
	
	private static Statements parseOptionalElse(List<ItemStack> code, AtomicInteger codeIndex) throws SyntaxException {
		
		ItemStack stack = getFirstNonNullItem(code, codeIndex);
		if(stack == null) {
			return null;
		}
		
		// Handle "else" Stmts.
		if(stack.getItem() == ModItems.ELSE.get()) {
			codeIndex.incrementAndGet();
			return Statements.parse(code, codeIndex, ModItems.END.get());
		}
		
		// Handle "elseif" Exp "then" Stmts IfElse.
		if(stack.getItem() == ModItems.ELSE_IF.get()) {
			int elseIfIndex = codeIndex.get();
			
			// Handle expression term.
			codeIndex.incrementAndGet();
			Expression condExp = Expression.parse(code, codeIndex);
			
			// Handle then term.
			ItemStack thenStack = getFirstNonNullItem(code, codeIndex);
			if(thenStack == null || thenStack.getItem() != ModItems.THEN.get()) {
				throw new ExpectedItemSyntaxException(ModItems.THEN.get(), codeIndex.get());
			}
			int thenIndex = codeIndex.get();
			
			// Handle statements term.
			codeIndex.incrementAndGet();
			Statements elseIfCodeStmts = Statements.parse(code, codeIndex,
					ModItems.END.get(), ModItems.ELSE_IF.get(),ModItems.ELSE.get());
			
			// Handle optional else terms.
			Statements elseCodeStmts = parseOptionalElse(code, codeIndex);
			
			// Rewrite this case to a new if statement that's placed within the else block of the other if statement.
			Statements ret = new Statements();
			int codeEndIndex = (elseCodeStmts != null ? elseCodeStmts.getCodeEndIndex()
					: (elseIfCodeStmts.getCodeEndIndex() != -1 ? elseIfCodeStmts.getCodeEndIndex() : thenIndex));
			ret.statements.add(new IfStatement(elseIfIndex, codeEndIndex, condExp, elseIfCodeStmts, elseCodeStmts));
			return ret;
		}
		
		// No optional else block found.
		return null;
	}
	
	@Override
	public ClassType typecheck(List<CompileException> compileExceptions) {
		this.requireType(this.condExp, ClassType.BOOLEAN, compileExceptions);
		this.ifCodeStmts.typecheck(compileExceptions);
		if(this.elseCodeStmts != null) {
			this.elseCodeStmts.typecheck(compileExceptions);
		}
		return ClassType.VOID;
	}
	
	@Override
	public String toCodeString() {
		return "if(" + this.condExp.toCodeString() + ") {\n\t"
				+ this.ifCodeStmts.toCodeString().replaceAll("\n", "\n\t") + "\n}" + (this.elseCodeStmts != null
				? " else {\n\t" + this.elseCodeStmts.toCodeString().replaceAll("\n", "\n\t") + "\n}" : "");
	}
}
