package computercraftsc.shared.turtle.core.code.ast;

import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

import computercraftsc.shared.RegistrySC.ModItems;
import computercraftsc.shared.turtle.core.code.compiler.exception.CompileException;
import computercraftsc.shared.turtle.core.code.compiler.exception.ExpectedItemSyntaxException;
import computercraftsc.shared.turtle.core.code.compiler.exception.SyntaxException;
import computercraftsc.shared.turtle.core.code.type.ClassType;
import net.minecraft.item.ItemStack;

public class WhileStatement extends Statement {
	
	public Expression condExp;
	public Statements codeStmts;
	
	public WhileStatement(int codeStartIndex, int codeEndIndex, Expression condExp, Statements codeStmts) {
		super(codeStartIndex, codeEndIndex);
		this.condExp = condExp;
		this.codeStmts = codeStmts;
	}
	
	public static WhileStatement parse(List<ItemStack> code, AtomicInteger codeIndex) throws SyntaxException {
		
		// Handle while term.
		ItemStack whileStack = getFirstNonNullItem(code, codeIndex);
		if(whileStack == null || whileStack.getItem() != ModItems.WHILE.get()) {
			throw new ExpectedItemSyntaxException(ModItems.WHILE.get(), codeIndex.get());
		}
		int whileIndex = codeIndex.get();
		
		// Handle expression term.
		codeIndex.incrementAndGet();
		Expression condExp = Expression.parse(code, codeIndex);
		
		// Handle do term.
		ItemStack doStack = getFirstNonNullItem(code, codeIndex);
		if(doStack == null || doStack.getItem() != ModItems.DO.get()) {
			throw new ExpectedItemSyntaxException(ModItems.DO.get(), codeIndex.get());
		}
		
		// Handle statements term.
		codeIndex.incrementAndGet();
		Statements codeStmts = Statements.parse(code, codeIndex, ModItems.END.get());
		
		// Handle end term.
		ItemStack endStack = getFirstNonNullItem(code, codeIndex);
		if(endStack == null || endStack.getItem() != ModItems.END.get()) {
			throw new ExpectedItemSyntaxException(ModItems.END.get(), codeIndex.get());
		}
		int endIndex = codeIndex.get();
		codeIndex.incrementAndGet();
		
		// Return the result.
		return new WhileStatement(whileIndex, endIndex, condExp, codeStmts);
	}
	
	@Override
	public String toCodeString() {
		return "while(" + this.condExp.toCodeString() + ") {\n\t"
				+ this.codeStmts.toCodeString().replaceAll("\n", "\n\t") + "\n}";
	}
	
	@Override
	public ClassType typecheck(List<CompileException> compileExceptions) {
		this.requireType(this.condExp, ClassType.BOOLEAN, compileExceptions);
		this.codeStmts.typecheck(compileExceptions);
		return ClassType.VOID;
	}
}
